<?php
/**
 * ================================================================================
 * Migration Script: Packagemaker → fixed_v2
 * البيانات التشغيلية الأساسية فقط (بدون limits و message_templates)
 * ================================================================================
 * 
 * Features:
 *  - Idempotent (يمكن تشغيله عدة مرات)
 *  - Resume capability (استئناف من آخر نقطة)
 *  - Dry-run mode (اختبار بدون تنفيذ)
 *  - Batching (معالجة دفعات)
 *  - Logging (سجلات تفصيلية)
 *  - Verification (تحقق تلقائي)
 * 
 * Usage:
 *  php migrate_core_no_templates.php                 # ترحيل عادي
 *  php migrate_core_no_templates.php --dry-run       # اختبار فقط
 *  php migrate_core_no_templates.php --with-sessions # مع control_sessions
 * 
 * Environment Variables (اختيارية):
 *  DB_HOST       - مضيف قاعدة البيانات (default: localhost)
 *  DB_USER       - اسم المستخدم (default: root)
 *  DB_PASS       - كلمة المرور (default: '')
 *  DB_SOURCE     - قاعدة المصدر (default: supehgku_Packagemaker)
 *  DB_TARGET     - قاعدة الهدف (default: supehgku_fixed_v2)
 * 
 * ================================================================================
 */

// ========================================
// Error Handling
// ========================================

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// ========================================
// Configuration - تحميل من ملفات الإعداد
// ========================================

// تحميل إعدادات قاعدة المصدر
if (!file_exists(__DIR__ . '/config_source.php')) {
    die("ERROR: ملف config_source.php غير موجود في نفس المجلد!\n");
}
if (!file_exists(__DIR__ . '/config_target.php')) {
    die("ERROR: ملف config_target.php غير موجود في نفس المجلد!\n");
}

$sourceConfig = require __DIR__ . '/config_source.php';
define('DB_SOURCE_HOST', $sourceConfig['DB_HOST']);
define('DB_SOURCE_NAME', $sourceConfig['DB_NAME']);
define('DB_SOURCE_USER', $sourceConfig['DB_USER']);
define('DB_SOURCE_PASS', $sourceConfig['DB_PASS']);

// تحميل إعدادات قاعدة الهدف
$targetConfig = require __DIR__ . '/config_target.php';
define('DB_TARGET_HOST', $targetConfig['DB_HOST']);
define('DB_TARGET_NAME', $targetConfig['DB_NAME']);
define('DB_TARGET_USER', $targetConfig['DB_USER']);
define('DB_TARGET_PASS', $targetConfig['DB_PASS']);

define('STATE_FILE', __DIR__ . '/.migrate_state.json');
define('LOG_FILE', __DIR__ . '/migrate_execution.log');

// Batch sizes
define('BATCH_SUB_BOTS', 50);
define('BATCH_STEAM_ACCOUNTS', 100);
define('BATCH_ALLOWED_USERS', 500);
define('BATCH_STEAM_REQUESTS', 1000);
define('BATCH_CONTROL_SESSIONS', 100);

// ========================================
// Parse Arguments
// ========================================

$isDryRun = in_array('--dry-run', $argv);
$withSessions = in_array('--with-sessions', $argv);

// ========================================
// Helper Functions
// ========================================

function logMessage($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    echo $logEntry;
    file_put_contents(LOG_FILE, $logEntry, FILE_APPEND);
}

function loadState() {
    if (!file_exists(STATE_FILE)) {
        return [
            'sub_bots_last_id' => 0,
            'steam_accounts_last_id' => 0,
            'allowed_users_last_id' => 0,
            'steam_requests_last_id' => 0,
            'control_sessions_last_id' => 0
        ];
    }
    
    $json = file_get_contents(STATE_FILE);
    return json_decode($json, true) ?: [];
}

function saveState($state) {
    file_put_contents(STATE_FILE, json_encode($state, JSON_PRETTY_PRINT));
}

function deleteState() {
    if (file_exists(STATE_FILE)) {
        unlink(STATE_FILE);
    }
}

function connectDB($type = 'source') {
    try {
        if ($type === 'source') {
            $dsn = "mysql:host=" . DB_SOURCE_HOST . ";dbname=" . DB_SOURCE_NAME . ";charset=utf8mb4";
            $pdo = new PDO($dsn, DB_SOURCE_USER, DB_SOURCE_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ]);
            logMessage("✅ Connected to SOURCE: " . DB_SOURCE_NAME, 'INFO');
        } else {
            $dsn = "mysql:host=" . DB_TARGET_HOST . ";dbname=" . DB_TARGET_NAME . ";charset=utf8mb4";
            $pdo = new PDO($dsn, DB_TARGET_USER, DB_TARGET_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ]);
            logMessage("✅ Connected to TARGET: " . DB_TARGET_NAME, 'INFO');
        }
        return $pdo;
    } catch (PDOException $e) {
        logMessage("Database connection failed: " . $e->getMessage(), 'ERROR');
        die("❌ Could not connect to database: $type\n");
    }
}

function getRowCount($pdo, $table) {
    $stmt = $pdo->query("SELECT COUNT(*) as cnt FROM `$table`");
    return $stmt->fetch()['cnt'];
}

// ========================================
// Migration Functions
// ========================================

function migrateSubBots($sourceDB, $targetDB, $state, $isDryRun) {
    logMessage("=== Starting sub_bots migration ===");
    
    $lastId = $state['sub_bots_last_id'];
    $batchSize = BATCH_SUB_BOTS;
    $totalMigrated = 0;
    
    $countStmt = $sourceDB->prepare("SELECT COUNT(*) as cnt FROM sub_bots WHERE id > ?");
    $countStmt->execute([$lastId]);
    $totalRemaining = $countStmt->fetch()['cnt'];
    
    if ($isDryRun) {
        logMessage("[DRY-RUN] Would migrate $totalRemaining sub_bots records", 'INFO');
        return $state;
    }
    
    while (true) {
        // Fetch batch
        $stmt = $sourceDB->prepare("
            SELECT id, label, bot_token, channel_id, max_users, 
                   welcome_message, is_paused, created_at, updated_at
            FROM sub_bots
            WHERE id > ?
            ORDER BY id ASC
            LIMIT ?
        ");
        $stmt->execute([$lastId, $batchSize]);
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            break;
        }
        
        // Insert batch
        $targetDB->beginTransaction();
        try {
            $insertStmt = $targetDB->prepare("
                INSERT INTO sub_bots 
                (id, label, bot_token, channel_id, max_users, welcome_message, is_paused, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    label = VALUES(label),
                    bot_token = VALUES(bot_token),
                    channel_id = VALUES(channel_id),
                    max_users = VALUES(max_users),
                    welcome_message = VALUES(welcome_message),
                    is_paused = VALUES(is_paused),
                    updated_at = VALUES(updated_at)
            ");
            
            foreach ($rows as $row) {
                $insertStmt->execute([
                    $row['id'], $row['label'], $row['bot_token'], $row['channel_id'],
                    $row['max_users'], $row['welcome_message'], $row['is_paused'],
                    $row['created_at'], $row['updated_at']
                ]);
                $lastId = $row['id'];
            }
            
            $targetDB->commit();
            $totalMigrated += count($rows);
            
            $state['sub_bots_last_id'] = $lastId;
            saveState($state);
            
            logMessage("Migrated batch: " . count($rows) . " records (last ID: $lastId)", 'INFO');
            
        } catch (PDOException $e) {
            $targetDB->rollBack();
            logMessage("Failed to migrate sub_bots batch: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    logMessage("✅ sub_bots migration completed: $totalMigrated records", 'SUCCESS');
    return $state;
}

function migrateSteamAccounts($sourceDB, $targetDB, $state, $isDryRun) {
    logMessage("=== Starting steam_accounts migration ===");
    
    $lastId = $state['steam_accounts_last_id'];
    $batchSize = BATCH_STEAM_ACCOUNTS;
    $totalMigrated = 0;
    
    $countStmt = $sourceDB->prepare("SELECT COUNT(*) as cnt FROM steam_accounts WHERE id > ?");
    $countStmt->execute([$lastId]);
    $totalRemaining = $countStmt->fetch()['cnt'];
    
    if ($isDryRun) {
        logMessage("[DRY-RUN] Would migrate $totalRemaining steam_accounts records", 'INFO');
        return $state;
    }
    
    while (true) {
        $stmt = $sourceDB->prepare("
            SELECT id, account_name, shared_secret, active, created_at, updated_at
            FROM steam_accounts
            WHERE id > ?
            ORDER BY id ASC
            LIMIT ?
        ");
        $stmt->execute([$lastId, $batchSize]);
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            break;
        }
        
        $targetDB->beginTransaction();
        try {
            $insertStmt = $targetDB->prepare("
                INSERT INTO steam_accounts 
                (id, account_name, shared_secret, active, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    account_name = VALUES(account_name),
                    shared_secret = VALUES(shared_secret),
                    active = VALUES(active),
                    updated_at = VALUES(updated_at)
            ");
            
            foreach ($rows as $row) {
                $insertStmt->execute([
                    $row['id'], $row['account_name'], $row['shared_secret'],
                    $row['active'], $row['created_at'], $row['updated_at']
                ]);
                $lastId = $row['id'];
            }
            
            $targetDB->commit();
            $totalMigrated += count($rows);
            
            $state['steam_accounts_last_id'] = $lastId;
            saveState($state);
            
            logMessage("Migrated batch: " . count($rows) . " records (last ID: $lastId)", 'INFO');
            
        } catch (PDOException $e) {
            $targetDB->rollBack();
            logMessage("Failed to migrate steam_accounts batch: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    logMessage("✅ steam_accounts migration completed: $totalMigrated records", 'SUCCESS');
    return $state;
}

function migrateAllowedUsers($sourceDB, $targetDB, $state, $isDryRun) {
    logMessage("=== Starting allowed_users migration ===");
    
    $lastId = $state['allowed_users_last_id'];
    $batchSize = BATCH_ALLOWED_USERS;
    $totalMigrated = 0;
    
    $countStmt = $sourceDB->prepare("SELECT COUNT(*) as cnt FROM allowed_users WHERE id > ?");
    $countStmt->execute([$lastId]);
    $totalRemaining = $countStmt->fetch()['cnt'];
    
    if ($isDryRun) {
        logMessage("[DRY-RUN] Would migrate $totalRemaining allowed_users records", 'INFO');
        return $state;
    }
    
    while (true) {
        $stmt = $sourceDB->prepare("
            SELECT id, bot_token, user_id, banned, created_at
            FROM allowed_users
            WHERE id > ?
            ORDER BY id ASC
            LIMIT ?
        ");
        $stmt->execute([$lastId, $batchSize]);
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            break;
        }
        
        $targetDB->beginTransaction();
        try {
            $insertStmt = $targetDB->prepare("
                INSERT INTO allowed_users 
                (id, bot_token, user_id, banned, created_at)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    banned = VALUES(banned),
                    created_at = VALUES(created_at)
            ");
            
            foreach ($rows as $row) {
                $insertStmt->execute([
                    $row['id'], $row['bot_token'], $row['user_id'],
                    $row['banned'], $row['created_at']
                ]);
                $lastId = $row['id'];
            }
            
            $targetDB->commit();
            $totalMigrated += count($rows);
            
            $state['allowed_users_last_id'] = $lastId;
            saveState($state);
            
            logMessage("Migrated batch: " . count($rows) . " records (last ID: $lastId)", 'INFO');
            
        } catch (PDOException $e) {
            $targetDB->rollBack();
            logMessage("Failed to migrate allowed_users batch: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    logMessage("✅ allowed_users migration completed: $totalMigrated records", 'SUCCESS');
    return $state;
}

function migrateSteamRequests($sourceDB, $targetDB, $state, $isDryRun) {
    logMessage("=== Starting steam_requests migration ===");
    
    $lastId = $state['steam_requests_last_id'];
    $batchSize = BATCH_STEAM_REQUESTS;
    $totalMigrated = 0;
    
    $countStmt = $sourceDB->prepare("SELECT COUNT(*) as cnt FROM steam_requests WHERE id > ?");
    $countStmt->execute([$lastId]);
    $totalRemaining = $countStmt->fetch()['cnt'];
    
    if ($isDryRun) {
        logMessage("[DRY-RUN] Would migrate $totalRemaining steam_requests records", 'INFO');
        return $state;
    }
    
    logMessage("Total steam_requests to migrate: $totalRemaining (this may take a while...)", 'INFO');
    
    while (true) {
        $stmt = $sourceDB->prepare("
            SELECT id, bot_token, user_id, account_name, request_time
            FROM steam_requests
            WHERE id > ?
            ORDER BY id ASC
            LIMIT ?
        ");
        $stmt->execute([$lastId, $batchSize]);
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            break;
        }
        
        $targetDB->beginTransaction();
        try {
            $insertStmt = $targetDB->prepare("
                INSERT INTO steam_requests 
                (id, bot_token, user_id, account_name, request_time)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    bot_token = VALUES(bot_token),
                    user_id = VALUES(user_id),
                    account_name = VALUES(account_name),
                    request_time = VALUES(request_time)
            ");
            
            foreach ($rows as $row) {
                $insertStmt->execute([
                    $row['id'], $row['bot_token'], $row['user_id'],
                    $row['account_name'], $row['request_time']
                ]);
                $lastId = $row['id'];
            }
            
            $targetDB->commit();
            $totalMigrated += count($rows);
            
            $state['steam_requests_last_id'] = $lastId;
            saveState($state);
            
            logMessage("Migrated batch: " . count($rows) . " records (last ID: $lastId, progress: $totalMigrated/$totalRemaining)", 'INFO');
            
        } catch (PDOException $e) {
            $targetDB->rollBack();
            logMessage("Failed to migrate steam_requests batch: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    logMessage("✅ steam_requests migration completed: $totalMigrated records", 'SUCCESS');
    return $state;
}

function migrateControlSessions($sourceDB, $targetDB, $state, $isDryRun) {
    logMessage("=== Starting control_sessions migration ===");
    
    $lastId = $state['control_sessions_last_id'];
    $batchSize = BATCH_CONTROL_SESSIONS;
    $totalMigrated = 0;
    
    $countStmt = $sourceDB->prepare("SELECT COUNT(*) as cnt FROM control_sessions WHERE id > ?");
    $countStmt->execute([$lastId]);
    $totalRemaining = $countStmt->fetch()['cnt'];
    
    if ($isDryRun) {
        logMessage("[DRY-RUN] Would migrate $totalRemaining control_sessions records", 'INFO');
        return $state;
    }
    
    while (true) {
        $stmt = $sourceDB->prepare("
            SELECT id, user_id, action, bot_token, created_at
            FROM control_sessions
            WHERE id > ?
            ORDER BY id ASC
            LIMIT ?
        ");
        $stmt->execute([$lastId, $batchSize]);
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            break;
        }
        
        $targetDB->beginTransaction();
        try {
            $insertStmt = $targetDB->prepare("
                INSERT INTO control_sessions 
                (id, user_id, action, bot_token, created_at)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    user_id = VALUES(user_id),
                    action = VALUES(action),
                    bot_token = VALUES(bot_token),
                    created_at = VALUES(created_at)
            ");
            
            foreach ($rows as $row) {
                $insertStmt->execute([
                    $row['id'], $row['user_id'], $row['action'],
                    $row['bot_token'], $row['created_at']
                ]);
                $lastId = $row['id'];
            }
            
            $targetDB->commit();
            $totalMigrated += count($rows);
            
            $state['control_sessions_last_id'] = $lastId;
            saveState($state);
            
            logMessage("Migrated batch: " . count($rows) . " records (last ID: $lastId)", 'INFO');
            
        } catch (PDOException $e) {
            $targetDB->rollBack();
            logMessage("Failed to migrate control_sessions batch: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    logMessage("✅ control_sessions migration completed: $totalMigrated records", 'SUCCESS');
    return $state;
}

function verifyMigration($sourceDB, $targetDB) {
    logMessage("=== Verifying migration ===");
    
    $tables = ['sub_bots', 'steam_accounts', 'allowed_users', 'steam_requests'];
    $allOk = true;
    
    foreach ($tables as $table) {
        $sourceCount = getRowCount($sourceDB, $table);
        $targetCount = getRowCount($targetDB, $table);
        
        $status = ($sourceCount <= $targetCount) ? '✅' : '❌';
        $allOk = $allOk && ($sourceCount <= $targetCount);
        
        logMessage("$status $table: Source=$sourceCount, Target=$targetCount", 'INFO');
    }
    
    // FK integrity checks
    logMessage("Checking Foreign Key integrity...", 'INFO');
    
    // Check allowed_users → sub_bots
    $stmt = $targetDB->query("
        SELECT COUNT(*) as cnt FROM allowed_users au
        LEFT JOIN sub_bots sb ON au.bot_token = sb.bot_token
        WHERE sb.bot_token IS NULL
    ");
    $orphaned = $stmt->fetch()['cnt'];
    logMessage(($orphaned == 0 ? '✅' : '❌') . " Orphaned allowed_users: $orphaned", 'INFO');
    $allOk = $allOk && ($orphaned == 0);
    
    // Check steam_requests → sub_bots
    $stmt = $targetDB->query("
        SELECT COUNT(*) as cnt FROM steam_requests sr
        LEFT JOIN sub_bots sb ON sr.bot_token = sb.bot_token
        WHERE sb.bot_token IS NULL
    ");
    $orphaned = $stmt->fetch()['cnt'];
    logMessage(($orphaned == 0 ? '✅' : '❌') . " Orphaned steam_requests (bot_token): $orphaned", 'INFO');
    $allOk = $allOk && ($orphaned == 0);
    
    // Check steam_requests → steam_accounts
    $stmt = $targetDB->query("
        SELECT COUNT(*) as cnt FROM steam_requests sr
        LEFT JOIN steam_accounts sa ON sr.account_name = sa.account_name
        WHERE sa.account_name IS NULL
    ");
    $orphaned = $stmt->fetch()['cnt'];
    logMessage(($orphaned == 0 ? '✅' : '❌') . " Orphaned steam_requests (account_name): $orphaned", 'INFO');
    $allOk = $allOk && ($orphaned == 0);
    
    return $allOk;
}

// ========================================
// Main Execution
// ========================================

try {
    logMessage("================================================================================");
    logMessage("Migration Script Started");
    logMessage("Mode: " . ($isDryRun ? 'DRY-RUN (Test Only)' : 'LIVE'));
    logMessage("With Sessions: " . ($withSessions ? 'YES' : 'NO'));
    logMessage("Source DB: " . DB_SOURCE_NAME);
    logMessage("Target DB: " . DB_TARGET_NAME);
    logMessage("================================================================================");
    
    // Connect to databases
    logMessage("Connecting to databases...");
    $sourceDB = connectDB('source');
    $targetDB = connectDB('target');
    
    // Load state
    $state = loadState();
    if ($state['sub_bots_last_id'] > 0) {
        logMessage("⚠️  Resuming from previous state: " . json_encode($state), 'WARN');
    }
    
    // Disable FK checks
    if (!$isDryRun) {
        $targetDB->exec("SET FOREIGN_KEY_CHECKS=0");
        logMessage("Foreign key checks disabled");
    }
    
    // Execute migrations
    $startTime = microtime(true);
    
    $state = migrateSubBots($sourceDB, $targetDB, $state, $isDryRun);
    $state = migrateSteamAccounts($sourceDB, $targetDB, $state, $isDryRun);
    $state = migrateAllowedUsers($sourceDB, $targetDB, $state, $isDryRun);
    $state = migrateSteamRequests($sourceDB, $targetDB, $state, $isDryRun);
    
    if ($withSessions && !$isDryRun) {
        $state = migrateControlSessions($sourceDB, $targetDB, $state, $isDryRun);
    } elseif ($withSessions && $isDryRun) {
        migrateControlSessions($sourceDB, $targetDB, $state, $isDryRun);
    }
    
    $endTime = microtime(true);
    $duration = round($endTime - $startTime, 2);
    
    // Re-enable FK checks
    if (!$isDryRun) {
        $targetDB->exec("SET FOREIGN_KEY_CHECKS=1");
        logMessage("Foreign key checks re-enabled");
    }
    
    // Verify
    if (!$isDryRun) {
        $verificationOk = verifyMigration($sourceDB, $targetDB);
        
        if ($verificationOk) {
            logMessage("================================================================================");
            logMessage("✅✅✅ MIGRATION COMPLETED SUCCESSFULLY! ✅✅✅", 'SUCCESS');
            logMessage("Duration: {$duration} seconds");
            logMessage("================================================================================");
            
            // Clean up state file
            deleteState();
            logMessage("State file cleaned up");
            
        } else {
            logMessage("⚠️ Migration completed with warnings - please review verification results", 'WARN');
        }
    } else {
        logMessage("================================================================================");
        logMessage("DRY-RUN completed successfully", 'INFO');
        logMessage("No data was modified. Run without --dry-run to execute migration.");
        logMessage("================================================================================");
    }
    
    logMessage("\nNext steps:");
    logMessage("1. Configure limits_global: UPDATE limits_global SET mode='weekly', per_day=2, weekly_cap=6, ban_days=7 WHERE id=1;");
    logMessage("2. Test one bot to ensure the system works");
    logMessage("3. Monitor system_logs for any issues");
    
} catch (Exception $e) {
    logMessage("❌❌❌ MIGRATION FAILED ❌❌❌", 'ERROR');
    logMessage("Error: " . $e->getMessage(), 'ERROR');
    logMessage("State saved. You can re-run the script to resume from the last successful point.", 'ERROR');
    exit(1);
}
?>
