<?php
/**
 * اختبار /account_details لتشخيص المشكلة
 */

declare(strict_types=1);

require __DIR__ . '/lib/telegram.php';

$cfg = env_cfg();
$CONTROL_TOKEN = (string)$cfg['CONTROL_BOT_TOKEN'];
$OWNER = (int)$cfg['OWNER_USER_ID'];

// تفعيل عرض الأخطاء
ini_set('display_errors', '1');
error_reporting(E_ALL);

echo "=== اختبار /account_details ===\n\n";

// الاتصال بقاعدة البيانات
try {
    $db = pdo();
    echo "✅ الاتصال بقاعدة البيانات: نجح\n";
} catch (Exception $e) {
    echo "❌ الاتصال بقاعدة البيانات: فشل\n";
    echo "الخطأ: " . $e->getMessage() . "\n";
    exit;
}

$search = 'Greatpackage1'; // اسم الحساب للاختبار

echo "\nالبحث عن الحساب: {$search}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

// البحث بالاسم
if (is_numeric($search)) {
    $stmt = $db->prepare("SELECT * FROM steam_accounts WHERE id=?");
    $stmt->execute([(int)$search]);
} else {
    $stmt = $db->prepare("SELECT * FROM steam_accounts WHERE account_name=?");
    $stmt->execute([$search]);
}

$account = $stmt->fetch();

if (!$account) {
    echo "❌ الحساب غير موجود\n";
    exit;
}

echo "✅ الحساب موجود:\n";
echo "   ID: {$account['id']}\n";
echo "   account_name: {$account['account_name']}\n";
echo "   active: {$account['active']}\n";
echo "   shared_secret: " . substr($account['shared_secret'], 0, 10) . "...\n";

// الحصول على المجموعة
$stmt = $db->prepare("
    SELECT ag.*
    FROM group_accounts ga
    JOIN account_groups ag ON ag.id = ga.group_id
    WHERE ga.account_id = ?
    LIMIT 1
");
$stmt->execute([$account['id']]);
$group = $stmt->fetch();

if ($group) {
    echo "\n✅ المجموعة موجودة:\n";
    echo "   ID: {$group['id']}\n";
    echo "   group_name: {$group['group_name']}\n";
} else {
    echo "\n⚠️ لا توجد مجموعة مرتبطة\n";
}

// الحصول على البوتات
$bots = [];
if ($group) {
    $stmt = $db->prepare("
        SELECT sb.id, sb.label, sb.is_paused
        FROM bot_groups bg
        JOIN sub_bots sb ON sb.id = bg.bot_id
        WHERE bg.group_id = ?
        ORDER BY sb.label ASC
    ");
    $stmt->execute([$group['id']]);
    $bots = $stmt->fetchAll();
    echo "\n✅ البوتات المرتبطة: " . count($bots) . "\n";
}

// الحصول على الحدود
$stmt = $db->prepare("SELECT * FROM limits_override WHERE account_id=?");
$stmt->execute([$account['id']]);
$override_limits = $stmt->fetch();

if ($override_limits) {
    echo "✅ حدود خاصة موجودة\n";
} else {
    echo "⚠️ لا توجد حدود خاصة\n";
}

// بناء الرسالة
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "بناء الرسالة:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$status = $account['active'] ? '✅ نشط' : '❌ معطل';
$acc_name_safe = htmlspecialchars($account['account_name'], ENT_QUOTES, 'UTF-8');

// عرض Steam Guard Secret بشكل آمن
$secret = $account['shared_secret'] ?? '';
if (!empty($secret)) {
    $secret_display = strlen($secret) > 10 ? substr($secret, 0, 10) . '...' : $secret;
} else {
    $secret_display = 'غير متوفر';
}
$secret_safe = htmlspecialchars($secret_display, ENT_QUOTES, 'UTF-8');

$out = "👤 <b>معلومات الحساب: {$acc_name_safe}</b>\n\n";
$out .= "🔹 <b>ID:</b> {$account['id']}\n";
$out .= "🔹 <b>الحالة:</b> {$status}\n";
$out .= "🔹 <b>Steam Guard:</b> <code>{$secret_safe}</code>\n\n";

// المجموعة
$out .= "📂 <b>المجموعة المرتبطة:</b>\n";
if ($group) {
    $status_icon = $group['active'] ? '✅' : '⛔';
    $group_name_safe = htmlspecialchars($group['group_name'], ENT_QUOTES, 'UTF-8');
    $out .= "   {$status_icon} <code>{$group_name_safe}</code> (ID: {$group['id']})\n";
} else {
    $out .= "   <i>لا توجد مجموعة مرتبطة</i>\n";
}

// البوتات
$out .= "\n🤖 <b>البوتات المرتبطة:</b> (" . count($bots) . ")\n";
if ($bots) {
    foreach ($bots as $i => $bot) {
        $num = $i + 1;
        $status_icon = $bot['is_paused'] ? '⏸️' : '▶️';
        $bot_label_safe = htmlspecialchars($bot['label'], ENT_QUOTES, 'UTF-8');
        $out .= "   {$num}. {$status_icon} <code>{$bot_label_safe}</code> (ID: {$bot['id']})\n";
    }
} else {
    $out .= "   <i>لا توجد بوتات مرتبطة</i>\n";
}

// الحدود
$out .= "\n⚙️ <b>الحدود الخاصة:</b>\n";
if ($override_limits) {
    $mode_ar = [
        'daily' => 'يومي',
        'weekly' => 'أسبوعي',
        'monthly' => 'شهري'
    ];
    $out .= "   ✅ <b>حدود خاصة بالحساب:</b>\n";
    $out .= "   • الوضع: " . ($mode_ar[$override_limits['mode']] ?? $override_limits['mode']) . "\n";
    $out .= "   • الحد اليومي: {$override_limits['per_day']} كود\n";
    if ($override_limits['cap']) $out .= "   • السقف: {$override_limits['cap']} كود\n";
    if ($override_limits['ban_days']) $out .= "   • مدة الحظر: {$override_limits['ban_days']} يوم\n";
} else {
    $out .= "   <i>يستخدم حدود البوت أو الحدود العامة</i>\n";
    $out .= "   <i>الأولوية: override → bot → global</i>\n";
}

echo "--- محتوى الرسالة ---\n";
echo $out;
echo "\n--- نهاية الرسالة ---\n";

echo "\nطول الرسالة: " . strlen($out) . " byte\n";
echo "عدد الأسطر: " . substr_count($out, "\n") . "\n";

// اختبار الإرسال
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "محاولة الإرسال إلى Telegram:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$params = [
    'chat_id' => $OWNER,
    'text' => $out,
    'parse_mode' => 'HTML',
    'disable_web_page_preview' => true
];

$result = tg_call($CONTROL_TOKEN, 'sendMessage', $params);

if ($result['ok'] ?? false) {
    echo "✅ تم الإرسال بنجاح!\n";
    echo "Message ID: " . ($result['result']['message_id'] ?? 'unknown') . "\n";
} else {
    echo "❌ فشل الإرسال!\n";
    echo "الخطأ: " . ($result['description'] ?? $result['error'] ?? 'unknown') . "\n";
    echo "رمز الخطأ: " . ($result['error_code'] ?? 'N/A') . "\n";
    
    if (isset($result['description'])) {
        echo "\nتحليل الخطأ:\n";
        if (strpos($result['description'], 'parse') !== false) {
            echo "⚠️ المشكلة: خطأ في HTML parsing\n";
            echo "   قد يكون بسبب أحرف خاصة في البيانات\n";
        }
        if (strpos($result['description'], 'too long') !== false) {
            echo "⚠️ المشكلة: الرسالة طويلة جداً (الحد الأقصى 4096)\n";
        }
    }
    
    echo "\nالرد الكامل من Telegram:\n";
    print_r($result);
}

echo "\n=== انتهى الاختبار ===\n";
