<?php
/**
 * ملف اختبار للأوامر الجديدة - لتشخيص المشكلة
 * 
 * استخدم هذا الملف لاختبار الأوامر وعرض الأخطاء
 */

declare(strict_types=1);

require __DIR__ . '/lib/telegram.php';

$cfg = env_cfg();
$CONTROL_TOKEN = (string)$cfg['CONTROL_BOT_TOKEN'];
$OWNER = (int)$cfg['OWNER_USER_ID'];

// تفعيل عرض الأخطاء
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

echo "=== بدء الاختبار ===\n\n";

// الاتصال بقاعدة البيانات
try {
    $db = pdo();
    echo "✅ الاتصال بقاعدة البيانات: نجح\n";
} catch (Exception $e) {
    echo "❌ الاتصال بقاعدة البيانات: فشل\n";
    echo "الخطأ: " . $e->getMessage() . "\n";
    exit;
}

// اختبار 1: جلب مجموعة
echo "\n--- اختبار 1: /group_details ---\n";
$group_name = 'default'; // غير هذا لاسم مجموعة موجودة

$stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
$stmt->execute([$group_name]);
$group = $stmt->fetch();

if (!$group) {
    echo "❌ المجموعة '{$group_name}' غير موجودة\n";
} else {
    echo "✅ المجموعة موجودة: ID = {$group['id']}\n";
    
    // جلب البوتات
    $stmt = $db->prepare("
        SELECT sb.id, sb.label, sb.is_paused
        FROM bot_groups bg
        JOIN sub_bots sb ON sb.id = bg.bot_id
        WHERE bg.group_id = ?
        ORDER BY sb.label ASC
    ");
    $stmt->execute([$group['id']]);
    $bots = $stmt->fetchAll();
    echo "   البوتات المرتبطة: " . count($bots) . "\n";
    
    // جلب الحسابات
    $stmt = $db->prepare("
        SELECT sa.id, sa.account_name, sa.active
        FROM group_accounts ga
        JOIN steam_accounts sa ON sa.id = ga.account_id
        WHERE ga.group_id = ?
        ORDER BY sa.account_name ASC
    ");
    $stmt->execute([$group['id']]);
    $accounts = $stmt->fetchAll();
    echo "   الحسابات المرتبطة: " . count($accounts) . "\n";
    
    // بناء الرسالة
    $status = $group['active'] ? '✅ نشطة' : '⛔ معطلة';
    $default = $group['is_default'] ? ' ⭐ (افتراضية)' : '';
    $group_name_safe = htmlspecialchars($group_name, ENT_QUOTES, 'UTF-8');
    $description_safe = htmlspecialchars($group['description'] ?: 'لا يوجد', ENT_QUOTES, 'UTF-8');
    
    $out = "📂 <b>معلومات المجموعة: {$group_name_safe}</b>{$default}\n\n";
    $out .= "🔹 <b>الحالة:</b> {$status}\n";
    $out .= "🔹 <b>الوصف:</b> {$description_safe}\n\n";
    
    // البوتات
    $out .= "🤖 <b>البوتات المرتبطة:</b> (" . count($bots) . ")\n";
    if ($bots) {
        foreach ($bots as $i => $bot) {
            $num = $i + 1;
            $status_icon = $bot['is_paused'] ? '⏸️' : '▶️';
            $bot_label_safe = htmlspecialchars($bot['label'], ENT_QUOTES, 'UTF-8');
            $out .= "{$num}. {$status_icon} <code>{$bot_label_safe}</code> (ID: {$bot['id']})\n";
        }
    } else {
        $out .= "   <i>لا توجد بوتات مرتبطة</i>\n";
    }
    
    $out .= "\n👤 <b>الحسابات المرتبطة:</b> (" . count($accounts) . ")\n";
    if ($accounts) {
        foreach ($accounts as $i => $acc) {
            $num = $i + 1;
            $status_icon = $acc['active'] ? '✅' : '❌';
            $acc_name_safe = htmlspecialchars($acc['account_name'], ENT_QUOTES, 'UTF-8');
            $out .= "{$num}. {$status_icon} <code>{$acc_name_safe}</code> (ID: {$acc['id']})\n";
        }
    } else {
        $out .= "   <i>لا توجد حسابات مرتبطة</i>\n";
    }
    
    echo "\n--- محتوى الرسالة ---\n";
    echo $out;
    echo "\n--- نهاية الرسالة ---\n";
    
    echo "\nطول الرسالة: " . strlen($out) . " byte\n";
    echo "عدد الأسطر: " . substr_count($out, "\n") . "\n";
    
    // اختبار الإرسال
    echo "\n--- محاولة الإرسال إلى Telegram ---\n";
    
    // استدعاء محسّن مع error logging
    $params = [
        'chat_id' => $OWNER,
        'text' => $out,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ];
    
    $result = tg_call($CONTROL_TOKEN, 'sendMessage', $params);
    
    if ($result['ok'] ?? false) {
        echo "✅ تم الإرسال بنجاح!\n";
        echo "Message ID: " . ($result['result']['message_id'] ?? 'unknown') . "\n";
    } else {
        echo "❌ فشل الإرسال!\n";
        echo "الخطأ: " . ($result['description'] ?? $result['error'] ?? 'unknown') . "\n";
        echo "رمز الخطأ: " . ($result['error_code'] ?? 'N/A') . "\n";
        
        if (isset($result['parameters'])) {
            echo "معلومات إضافية:\n";
            print_r($result['parameters']);
        }
        
        // تحليل الخطأ
        if (isset($result['description'])) {
            if (strpos($result['description'], 'parse') !== false) {
                echo "\n⚠️ المشكلة: خطأ في HTML parsing\n";
                echo "   الحل: تحقق من الأحرف الخاصة في البيانات\n";
            }
            if (strpos($result['description'], 'too long') !== false) {
                echo "\n⚠️ المشكلة: الرسالة طويلة جداً\n";
                echo "   الحد الأقصى: 4096 حرف\n";
            }
        }
    }
}

echo "\n=== انتهى الاختبار ===\n";
