<?php
/**
 * ===============================================================
 * نظام حدود البوتات - Bot-Specific Limits Implementation
 * ===============================================================
 * 
 * الأوامر الجديدة لدعم حدود خاصة لكل بوت
 * الأولوية: override → bot → global
 * 
 * النسخة: 2.0
 * التاريخ: 2025-10-25
 */

// ====== دالة الحصول على الحدود المطبقة ======
/**
 * الحصول على الحدود المطبقة حسب الأولوية
 * 
 * @param PDO $pdo
 * @param string $bot_label اسم البوت
 * @param int|null $account_id معرف الحساب (اختياري)
 * @return array ['mode', 'per_day', 'weekly_cap', 'monthly_cap', 'ban_days', 'source']
 */
function getAppliedLimits($pdo, $bot_label, $account_id = null) {
    // 1️⃣ التحقق من override (أعلى أولوية)
    if ($account_id) {
        $stmt = $pdo->prepare("SELECT * FROM limits_override WHERE account_id = ?");
        $stmt->execute([$account_id]);
        $override = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($override) {
            return [
                'mode' => $override['mode'],
                'per_day' => $override['per_day'],
                'weekly_cap' => $override['mode'] === 'weekly' ? $override['cap'] : null,
                'monthly_cap' => $override['mode'] === 'monthly' ? $override['cap'] : null,
                'ban_days' => $override['ban_days'],
                'source' => 'override'
            ];
        }
    }
    
    // 2️⃣ التحقق من حدود البوت الخاصة
    $stmt = $pdo->prepare("SELECT * FROM limits_bot WHERE bot_label = ?");
    $stmt->execute([$bot_label]);
    $bot_limit = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($bot_limit) {
        return [
            'mode' => $bot_limit['mode'],
            'per_day' => $bot_limit['per_day'],
            'weekly_cap' => $bot_limit['weekly_cap'],
            'monthly_cap' => $bot_limit['monthly_cap'],
            'ban_days' => $bot_limit['ban_days'],
            'source' => 'bot'
        ];
    }
    
    // 3️⃣ استخدام الحدود العامة (افتراضي)
    $stmt = $pdo->query("SELECT * FROM limits_global WHERE id = 1");
    $global = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return [
        'mode' => $global['mode'],
        'per_day' => $global['per_day'],
        'weekly_cap' => $global['weekly_cap'],
        'monthly_cap' => $global['monthly_cap'],
        'ban_days' => $global['ban_days'],
        'source' => 'global'
    ];
}

// ====== معالج الأوامر الجديدة ======

switch ($command) {
    
    // ========================================
    // تغيير وضع البوت
    // ========================================
    case '/limit_mode':
        if (count($parts) < 3) {
            return "❌ <b>الاستخدام:</b>\n<code>/limit_mode &lt;bot_label&gt; &lt;mode&gt;</code>\n\n" .
                   "📌 <b>الأوضاع:</b>\n" .
                   "• <code>daily</code> - يومي\n" .
                   "• <code>weekly</code> - أسبوعي\n" .
                   "• <code>monthly</code> - شهري";
        }
        
        $bot_label = $parts[1];
        $mode = strtolower($parts[2]);
        
        if (!in_array($mode, ['daily', 'weekly', 'monthly'])) {
            return "❌ الوضع غير صحيح!\n\nاستخدم: <code>daily</code>, <code>weekly</code>, <code>monthly</code>";
        }
        
        // التحقق من وجود البوت
        $check = $pdo->prepare("SELECT id FROM sub_bots WHERE label = ?");
        $check->execute([$bot_label]);
        if (!$check->fetch()) {
            return "❌ البوت '<b>$bot_label</b>' غير موجود!\n\n💡 لعرض البوتات: <code>/list</code>";
        }
        
        // تحديث أو إدراج
        $stmt = $pdo->prepare("
            INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) 
            VALUES (?, ?, 2, NULL, NULL, NULL)
            ON DUPLICATE KEY UPDATE mode = VALUES(mode)
        ");
        
        $stmt->execute([$bot_label, $mode]);
        
        $mode_name = ['daily' => 'يومي', 'weekly' => 'أسبوعي', 'monthly' => 'شهري'][$mode];
        
        return "✅ تم تعيين وضع '<b>$mode_name</b>' للبوت '<b>$bot_label</b>'\n\n" .
               "💡 لتخصيص الحدود: <code>/set_$mode $bot_label ...</code>";
    
    // ========================================
    // تعيين حد يومي لبوت
    // ========================================
    case '/set_daily':
        if (count($parts) < 3) {
            return "❌ <b>الاستخدام:</b>\n<code>/set_daily &lt;bot_label&gt; &lt;per_day&gt;</code>\n\n" .
                   "مثال: <code>/set_daily testbot 3</code>\n" .
                   "→ 3 أكواد يومياً، رفض فوري بعدها";
        }
        
        $bot_label = $parts[1];
        $per_day = (int)$parts[2];
        
        if ($per_day < 1) {
            return "❌ الحد اليومي يجب أن يكون 1 على الأقل";
        }
        
        // التحقق من وجود البوت
        $check = $pdo->prepare("SELECT id FROM sub_bots WHERE label = ?");
        $check->execute([$bot_label]);
        if (!$check->fetch()) {
            return "❌ البوت '<b>$bot_label</b>' غير موجود!";
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) 
            VALUES (?, 'daily', ?, NULL, NULL, NULL)
            ON DUPLICATE KEY UPDATE 
                mode = 'daily', 
                per_day = VALUES(per_day),
                weekly_cap = NULL,
                monthly_cap = NULL,
                ban_days = NULL
        ");
        
        $stmt->execute([$bot_label, $per_day]);
        
        return "✅ <b>تم تعيين حد يومي للبوت '$bot_label'</b>\n\n" .
               "📊 الحدود:\n" .
               "• 📅 <b>يومي:</b> $per_day كود\n" .
               "• 🚫 <b>الحظر:</b> بدون حظر (رفض فوري)\n\n" .
               "💡 للتحقق: <code>/show_limit bot/$bot_label</code>";
    
    // ========================================
    // تعيين حد أسبوعي لبوت
    // ========================================
    case '/set_weekly':
        if (count($parts) < 5) {
            return "❌ <b>الاستخدام:</b>\n<code>/set_weekly &lt;bot&gt; &lt;per_day&gt; &lt;weekly_cap&gt; &lt;ban_days&gt;</code>\n\n" .
                   "مثال: <code>/set_weekly vipbot 5 20 3</code>\n" .
                   "→ 5 كود/يوم، 20/أسبوع، حظر 3 أيام";
        }
        
        $bot_label = $parts[1];
        $per_day = (int)$parts[2];
        $weekly_cap = (int)$parts[3];
        $ban_days = (int)$parts[4];
        
        if ($per_day < 1) {
            return "❌ الحد اليومي يجب أن يكون 1 على الأقل";
        }
        
        if ($weekly_cap < $per_day) {
            return "❌ الحد الأسبوعي (<b>$weekly_cap</b>) يجب أن يكون أكبر من أو يساوي الحد اليومي (<b>$per_day</b>)";
        }
        
        // التحقق من وجود البوت
        $check = $pdo->prepare("SELECT id FROM sub_bots WHERE label = ?");
        $check->execute([$bot_label]);
        if (!$check->fetch()) {
            return "❌ البوت '<b>$bot_label</b>' غير موجود!";
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) 
            VALUES (?, 'weekly', ?, ?, NULL, ?)
            ON DUPLICATE KEY UPDATE 
                mode = 'weekly',
                per_day = VALUES(per_day),
                weekly_cap = VALUES(weekly_cap),
                monthly_cap = NULL,
                ban_days = VALUES(ban_days)
        ");
        
        $stmt->execute([$bot_label, $per_day, $weekly_cap, $ban_days]);
        
        return "✅ <b>تم تعيين حد أسبوعي للبوت '$bot_label'</b>\n\n" .
               "📊 الحدود:\n" .
               "• 📅 <b>يومي:</b> $per_day كود\n" .
               "• 📆 <b>أسبوعي:</b> $weekly_cap كود\n" .
               "• 🚫 <b>حظر:</b> $ban_days أيام\n\n" .
               "💡 للتحقق: <code>/show_limit bot/$bot_label</code>";
    
    // ========================================
    // تعيين حد شهري لبوت
    // ========================================
    case '/set_monthly':
        if (count($parts) < 5) {
            return "❌ <b>الاستخدام:</b>\n<code>/set_monthly &lt;bot&gt; &lt;per_day&gt; &lt;monthly_cap&gt; &lt;ban_days&gt;</code>\n\n" .
                   "مثال: <code>/set_monthly premiumbot 10 100 7</code>\n" .
                   "→ 10 كود/يوم، 100/شهر، حظر 7 أيام";
        }
        
        $bot_label = $parts[1];
        $per_day = (int)$parts[2];
        $monthly_cap = (int)$parts[3];
        $ban_days = (int)$parts[4];
        
        if ($per_day < 1) {
            return "❌ الحد اليومي يجب أن يكون 1 على الأقل";
        }
        
        if ($monthly_cap < $per_day) {
            return "❌ الحد الشهري (<b>$monthly_cap</b>) يجب أن يكون أكبر من أو يساوي الحد اليومي (<b>$per_day</b>)";
        }
        
        // التحقق من وجود البوت
        $check = $pdo->prepare("SELECT id FROM sub_bots WHERE label = ?");
        $check->execute([$bot_label]);
        if (!$check->fetch()) {
            return "❌ البوت '<b>$bot_label</b>' غير موجود!";
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) 
            VALUES (?, 'monthly', ?, NULL, ?, ?)
            ON DUPLICATE KEY UPDATE 
                mode = 'monthly',
                per_day = VALUES(per_day),
                weekly_cap = NULL,
                monthly_cap = VALUES(monthly_cap),
                ban_days = VALUES(ban_days)
        ");
        
        $stmt->execute([$bot_label, $per_day, $monthly_cap, $ban_days]);
        
        return "✅ <b>تم تعيين حد شهري للبوت '$bot_label'</b>\n\n" .
               "📊 الحدود:\n" .
               "• 📅 <b>يومي:</b> $per_day كود\n" .
               "• 📆 <b>شهري:</b> $monthly_cap كود\n" .
               "• 🚫 <b>حظر:</b> $ban_days أيام\n\n" .
               "💡 للتحقق: <code>/show_limit bot/$bot_label</code>";
    
    // ========================================
    // حذف حدود البوت
    // ========================================
    case '/bot_limit_clear':
        if (count($parts) < 2) {
            return "❌ <b>الاستخدام:</b>\n<code>/bot_limit_clear &lt;bot_label&gt;</code>\n\n" .
                   "مثال: <code>/bot_limit_clear testbot</code>";
        }
        
        $bot_label = $parts[1];
        
        $stmt = $pdo->prepare("DELETE FROM limits_bot WHERE bot_label = ?");
        $stmt->execute([$bot_label]);
        
        if ($stmt->rowCount() > 0) {
            return "✅ تم حذف الحدود الخاصة للبوت '<b>$bot_label</b>'\n\n" .
                   "🌍 سيستخدم البوت الآن الحدود العامة\n\n" .
                   "💡 للتحقق: <code>/show_limit bot/$bot_label</code>";
        } else {
            return "ℹ️ البوت '<b>$bot_label</b>' لم يكن لديه حدود خاصة\n\n" .
                   "🌍 يستخدم الحدود العامة حالياً";
        }
    
    // ========================================
    // عرض الحدود (محدّث)
    // ========================================
    case '/show_limit':
        if (count($parts) < 2) {
            return "❌ <b>الاستخدام:</b>\n\n" .
                   "<code>/show_limit global</code> - الحدود العامة\n" .
                   "<code>/show_limit bot</code> - جميع البوتات\n" .
                   "<code>/show_limit bot/&lt;label&gt;</code> - بوت محدد\n" .
                   "<code>/show_limit account/&lt;id&gt;</code> - حساب محدد\n\n" .
                   "مثال: <code>/show_limit bot/vipbot</code>";
        }
        
        $target = $parts[1];
        
        // --- عرض الحدود العامة ---
        if ($target === 'global') {
            $stmt = $pdo->query("SELECT * FROM limits_global WHERE id = 1");
            $limit = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$limit) {
                return "❌ لا توجد حدود عامة محددة!";
            }
            
            $mode_name = [
                'daily' => '📅 يومي',
                'weekly' => '📆 أسبوعي',
                'monthly' => '📆 شهري'
            ][$limit['mode']];
            
            $msg = "🌍 <b>الحدود العامة (Global Limits)</b>\n\n";
            $msg .= "📊 الوضع: $mode_name\n";
            $msg .= "📅 الحد اليومي: <code>{$limit['per_day']}</code> كود\n";
            
            if ($limit['mode'] === 'weekly' && $limit['weekly_cap']) {
                $msg .= "📆 السقف الأسبوعي: <code>{$limit['weekly_cap']}</code> كود\n";
            }
            
            if ($limit['mode'] === 'monthly' && $limit['monthly_cap']) {
                $msg .= "📆 السقف الشهري: <code>{$limit['monthly_cap']}</code> كود\n";
            }
            
            if ($limit['ban_days']) {
                $msg .= "🚫 مدة الحظر: <code>{$limit['ban_days']}</code> أيام\n";
            } else {
                $msg .= "🚫 مدة الحظر: <code>بدون حظر</code>\n";
            }
            
            $msg .= "\n💡 <i>هذه الحدود تُطبق على جميع البوتات ما لم يكن لديها حدود خاصة</i>";
            
            return $msg;
        }
        
        // --- عرض حدود جميع البوتات ---
        if ($target === 'bot') {
            $stmt = $pdo->query("
                SELECT 
                    sb.label,
                    lb.mode,
                    lb.per_day,
                    lb.weekly_cap,
                    lb.monthly_cap,
                    lb.ban_days,
                    CASE WHEN lb.bot_label IS NOT NULL THEN 1 ELSE 0 END as has_custom
                FROM sub_bots sb
                LEFT JOIN limits_bot lb ON lb.bot_label = sb.label
                ORDER BY has_custom DESC, sb.label
            ");
            
            $bots = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($bots)) {
                return "❌ لا توجد بوتات مسجلة!";
            }
            
            $msg = "🤖 <b>حدود البوتات (Bot Limits)</b>\n\n";
            
            $with_custom = array_filter($bots, fn($b) => $b['has_custom']);
            $without_custom = array_filter($bots, fn($b) => !$b['has_custom']);
            
            if (!empty($with_custom)) {
                $msg .= "✅ <b>بوتات لديها حدود خاصة:</b>\n\n";
                
                foreach ($with_custom as $bot) {
                    $mode_emoji = [
                        'daily' => '📅',
                        'weekly' => '📆',
                        'monthly' => '📆'
                    ][$bot['mode']];
                    
                    $mode_name = [
                        'daily' => 'يومي',
                        'weekly' => 'أسبوعي',
                        'monthly' => 'شهري'
                    ][$bot['mode']];
                    
                    $msg .= "• <b>{$bot['label']}</b>\n";
                    $msg .= "  $mode_emoji $mode_name: {$bot['per_day']} كود/يوم";
                    
                    if ($bot['mode'] === 'weekly' && $bot['weekly_cap']) {
                        $msg .= ", {$bot['weekly_cap']}/أسبوع";
                    }
                    if ($bot['mode'] === 'monthly' && $bot['monthly_cap']) {
                        $msg .= ", {$bot['monthly_cap']}/شهر";
                    }
                    if ($bot['ban_days']) {
                        $msg .= ", 🚫 {$bot['ban_days']} أيام";
                    }
                    
                    $msg .= "\n";
                }
            }
            
            if (!empty($without_custom)) {
                $msg .= "\n🌍 <b>بوتات تستخدم الحدود العامة:</b>\n";
                foreach ($without_custom as $bot) {
                    $msg .= "• {$bot['label']}\n";
                }
                $msg .= "\n💡 <i>لتخصيص حدود: /set_weekly bot per_day cap ban</i>";
            }
            
            return $msg;
        }
        
        // --- عرض حدود بوت محدد ---
        if (strpos($target, 'bot/') === 0) {
            $bot_label = substr($target, 4);
            
            // التحقق من وجود البوت
            $check = $pdo->prepare("SELECT id FROM sub_bots WHERE label = ?");
            $check->execute([$bot_label]);
            if (!$check->fetch()) {
                return "❌ البوت '<b>$bot_label</b>' غير موجود!";
            }
            
            // جلب الحدود الخاصة
            $stmt = $pdo->prepare("SELECT * FROM limits_bot WHERE bot_label = ?");
            $stmt->execute([$bot_label]);
            $bot_limit = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $msg = "🤖 <b>حدود البوت: $bot_label</b>\n\n";
            
            if ($bot_limit) {
                $mode_name = [
                    'daily' => 'يومي',
                    'weekly' => 'أسبوعي',
                    'monthly' => 'شهري'
                ][$bot_limit['mode']];
                
                $msg .= "✅ <b>حدود خاصة:</b>\n\n";
                $msg .= "📊 الوضع: <code>$mode_name</code>\n";
                $msg .= "📅 الحد اليومي: <code>{$bot_limit['per_day']}</code> كود\n";
                
                if ($bot_limit['mode'] === 'weekly' && $bot_limit['weekly_cap']) {
                    $msg .= "📆 السقف الأسبوعي: <code>{$bot_limit['weekly_cap']}</code> كود\n";
                }
                
                if ($bot_limit['mode'] === 'monthly' && $bot_limit['monthly_cap']) {
                    $msg .= "📆 السقف الشهري: <code>{$bot_limit['monthly_cap']}</code> كود\n";
                }
                
                if ($bot_limit['ban_days']) {
                    $msg .= "🚫 مدة الحظر: <code>{$bot_limit['ban_days']}</code> أيام\n";
                } else {
                    $msg .= "🚫 مدة الحظر: <code>بدون حظر</code>\n";
                }
                
                $msg .= "\n💡 <i>لحذف الحدود الخاصة: </i><code>/bot_limit_clear $bot_label</code>";
                
            } else {
                // جلب الحدود العامة
                $stmt = $pdo->query("SELECT * FROM limits_global WHERE id = 1");
                $global = $stmt->fetch(PDO::FETCH_ASSOC);
                
                $mode_name = [
                    'daily' => 'يومي',
                    'weekly' => 'أسبوعي',
                    'monthly' => 'شهري'
                ][$global['mode']];
                
                $msg .= "🌍 <b>يستخدم الحدود العامة:</b>\n\n";
                $msg .= "📊 الوضع: <code>$mode_name</code>\n";
                $msg .= "📅 الحد اليومي: <code>{$global['per_day']}</code> كود\n";
                
                if ($global['mode'] === 'weekly' && $global['weekly_cap']) {
                    $msg .= "📆 السقف الأسبوعي: <code>{$global['weekly_cap']}</code> كود\n";
                }
                
                if ($global['mode'] === 'monthly' && $global['monthly_cap']) {
                    $msg .= "📆 السقف الشهري: <code>{$global['monthly_cap']}</code> كود\n";
                }
                
                if ($global['ban_days']) {
                    $msg .= "🚫 مدة الحظر: <code>{$global['ban_days']}</code> أيام\n";
                }
                
                $msg .= "\n💡 <i>لتخصيص حدود لهذا البوت: </i><code>/set_weekly $bot_label per_day cap ban</code>";
            }
            
            return $msg;
        }
        
        // --- عرض حدود حساب محدد ---
        if (strpos($target, 'account/') === 0) {
            $account_id = (int)substr($target, 8);
            
            // التحقق من وجود الحساب
            $check = $pdo->prepare("SELECT account_name FROM steam_accounts WHERE id = ?");
            $check->execute([$account_id]);
            $account = $check->fetch();
            
            if (!$account) {
                return "❌ الحساب #$account_id غير موجود!";
            }
            
            // جلب التجاوز
            $stmt = $pdo->prepare("SELECT * FROM limits_override WHERE account_id = ?");
            $stmt->execute([$account_id]);
            $override = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $msg = "🎮 <b>حدود الحساب: {$account['account_name']}</b>\n";
            $msg .= "🆔 ID: <code>$account_id</code>\n\n";
            
            if ($override) {
                $mode_name = [
                    'daily' => 'يومي',
                    'weekly' => 'أسبوعي',
                    'monthly' => 'شهري'
                ][$override['mode']];
                
                $msg .= "⭐ <b>حدود مخصصة (Override):</b>\n\n";
                $msg .= "📊 الوضع: <code>$mode_name</code>\n";
                $msg .= "📅 الحد اليومي: <code>{$override['per_day']}</code> كود\n";
                
                if ($override['cap']) {
                    if ($override['mode'] === 'weekly') {
                        $msg .= "📆 السقف الأسبوعي: <code>{$override['cap']}</code> كود\n";
                    } else {
                        $msg .= "📆 السقف الشهري: <code>{$override['cap']}</code> كود\n";
                    }
                }
                
                if ($override['ban_days']) {
                    $msg .= "🚫 مدة الحظر: <code>{$override['ban_days']}</code> أيام\n";
                }
                
                $msg .= "\n💡 <i>لحذف التجاوز: </i><code>/override_clear $account_id</code>";
                
            } else {
                $msg .= "🌍 <b>يستخدم حدود البوت أو العامة</b>\n\n";
                $msg .= "💡 <i>لتخصيص حدود: </i><code>/override_account $account_id mode per_day cap ban</code>";
            }
            
            return $msg;
        }
        
        return "❌ صيغة غير صحيحة!\n\n" .
               "<b>الاستخدام:</b>\n" .
               "<code>/show_limit global</code>\n" .
               "<code>/show_limit bot</code>\n" .
               "<code>/show_limit bot/&lt;label&gt;</code>\n" .
               "<code>/show_limit account/&lt;id&gt;</code>";
}
