# تقرير قاعدة البيانات (Database Schema)

## 📊 نظرة عامة

**اسم قاعدة البيانات**: `supehgku_botsteampro`

**عدد الجداول**: 34+ جدول

**عدد الـ Views**: 2 (active_bans, current_usage)

**عدد الـ Procedures**: 2 (cleanup_expired_bans, reset_expired_counters)

**المحرك**: InnoDB (يدعم Transactions و Foreign Keys)

**الترميز**: utf8mb4_unicode_ci

---

## 🗂️ تصنيف الجداول

### 1️⃣ إدارة البوتات (4 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `sub_bots` | البوتات الفرعية المسجلة | label, bot_token, channel_id, max_users, is_paused |
| `bot_groups` | ربط البوتات بالمجموعات | bot_id, group_id |
| `bot_managers` | المدراء الفرعيون للبوتات | user_id, bot_token, permissions, is_active |
| `bot_message_templates` | قوالب الرسائل الخاصة بكل بوت | bot_id, template_key, template_text |

### 2️⃣ إدارة المستخدمين (3 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `allowed_users` | المستخدمون المسموح لهم | bot_token, user_id, banned |
| `control_sessions` | جلسات مؤقتة (await_welcome) | user_id, action, bot_token |
| `steam_requests` | سجل طلبات الأكواد | bot_token, user_id, account_name, request_time |

### 3️⃣ الحسابات والمجموعات (3 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `steam_accounts` | حسابات Steam Guard | account_name, shared_secret, active |
| `account_groups` | مجموعات الحسابات | group_name, is_default, active, priority |
| `group_accounts` | ربط الحسابات بالمجموعات | group_id, account_id |

### 4️⃣ نظام الحدود (6 جداول + 2 views)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `limits_global` | الحدود العامة للجميع | mode, per_day, weekly_cap, monthly_cap, ban_days |
| `limits_bot` | حدود خاصة ببوت معين | bot_label, mode, per_day, weekly_cap, monthly_cap |
| `limits_override` | تجاوز حدود لحساب معين | account_id, mode, per_day, cap, ban_days |
| `limit_counters` | عدادات الاستخدام | user_id, account_id, bot_label, day_used, week_used, month_used |
| `limit_bans` | الحظر من حساب معين | user_id, account_id, bot_label, banned_until, reason |
| `limit_messages` | رسائل أحداث الحدود | event, text |
| **VIEW: `active_bans`** | الحظورات النشطة | id, user_id, account_id, banned_until, hours_remaining |
| **VIEW: `current_usage`** | الاستخدام الحالي | user_id, account_id, day_used, week_used, month_used |

### 5️⃣ نظام القسائم (3 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `vouchers` | القسائم المتاحة | code, voucher_type, amount, target_label, max_uses, current_uses |
| `voucher_redemptions` | سجل استخدام القسائم | voucher_code, bot_token, redeemed_by, amount_added |
| `voucher_usage` | استخدامات القسائم (قديم) | voucher_code, bot_id, redeemer_id |

### 6️⃣ نظام البث (2 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `broadcast_jobs` | مهام البث (لمنع التكرار) | job_key, initiator_user_id, target_type, message_text, status |
| `broadcast_log` | سجل عمليات البث | bot_token, manager_id, message_text, recipients_count |

### 7️⃣ القوالب والرسائل (2 جداول)

| الجدول | الوظيفة | الحقول الرئيسية |
|--------|---------|-----------------|
| `message_templates` | قوالب الرسائل العامة | key, text |
| `system_logs` | سجل النظام | log_level, component, message, created_at |

---

## 🔗 العلاقات بين الجداول (Relationships)

### 1. البوتات والمستخدمين

```
sub_bots (1) ──────< (N) allowed_users
   │                       │
   │                       └─ user_id (Telegram User)
   └─ bot_token (UNIQUE)

Foreign Key: allowed_users.bot_token → sub_bots.bot_token
  ON DELETE CASCADE (حذف البوت = حذف جميع مستخدميه)
```

### 2. البوتات والمجموعات

```
sub_bots (1) ──────< (N) bot_groups (N) ──────> (1) account_groups
   │                        │                           │
   id                    bot_id, group_id             id
                                                       │
                                                       │
account_groups (1) ──────< (N) group_accounts (N) ──────> (1) steam_accounts
                              │                              │
                           group_id, account_id            id

Foreign Keys:
- bot_groups.bot_id → sub_bots.id (ON DELETE CASCADE)
- bot_groups.group_id → account_groups.id (ON DELETE CASCADE)
- group_accounts.group_id → account_groups.id (ON DELETE CASCADE)
- group_accounts.account_id → steam_accounts.id (ON DELETE CASCADE)
```

### 3. المدراء والبوتات

```
bot_managers
   │
   ├─ user_id (Telegram User ID)
   └─ bot_token (NULL = مدير عام لكل البوتات)
       │
       └──> sub_bots.bot_token

Foreign Key: bot_managers.bot_token → sub_bots.bot_token
  ON DELETE CASCADE (حذف البوت = حذف مدرائه)
```

### 4. نظام الحدود

```
limit_counters
   │
   ├─ user_id (Telegram User)
   ├─ account_id ──────> steam_accounts.id
   └─ bot_label (لجعل العدادات مستقلة لكل بوت)

limit_bans
   │
   ├─ user_id (Telegram User)
   ├─ account_id ──────> steam_accounts.id
   └─ bot_label

limits_override
   │
   └─ account_id ──────> steam_accounts.id (UNIQUE)

limits_bot
   │
   └─ bot_label ──────> sub_bots.label (UNIQUE)
```

### 5. القسائم

```
vouchers
   │
   ├─ code (UNIQUE)
   ├─ target_label (NULL = لجميع البوتات)
   └─ created_by_user_id (المالك)

voucher_redemptions
   │
   ├─ voucher_code ──────> vouchers.code
   ├─ bot_token ──────> sub_bots.bot_token
   └─ redeemed_by (user_id الذي استخدم القسيمة)

Foreign Key: voucher_redemptions.bot_token → sub_bots.bot_token
  ON DELETE CASCADE
```

---

## 📋 الجداول بالتفصيل

### 1. جدول `sub_bots` (البوتات الفرعية)

```sql
CREATE TABLE `sub_bots` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `label` varchar(64) UNIQUE NOT NULL,
  `bot_token` varchar(128) UNIQUE NOT NULL,
  `channel_id` varchar(64) DEFAULT NULL,
  `max_users` int(11) DEFAULT 20,
  `daily_attempts` int(11) DEFAULT 5,  -- (مهمل - لم يعد يُستخدم)
  `is_paused` tinyint(1) DEFAULT 0,
  `welcome_message` text DEFAULT NULL,
  `created_at` timestamp DEFAULT current_timestamp()
);
```

**الفهارس**:
- PRIMARY KEY: `id`
- UNIQUE: `label`, `bot_token`

**الوظيفة**: تخزين بيانات البوتات الفرعية

**ملاحظات**:
- `label`: اسم قصير مفهوم للبوت (test1, main, etc)
- `daily_attempts`: قديم ولم يعد يُستخدم (تم استبداله بنظام الحدود الذكي)

---

### 2. جدول `allowed_users` (المستخدمون المسموح لهم)

```sql
CREATE TABLE `allowed_users` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `bot_token` varchar(128) NOT NULL,
  `user_id` bigint(20) NOT NULL,
  `banned` tinyint(1) DEFAULT 0,
  `created_at` timestamp DEFAULT current_timestamp(),
  UNIQUE KEY (`bot_token`, `user_id`),
  FOREIGN KEY (`bot_token`) REFERENCES `sub_bots`(`bot_token`) 
    ON DELETE CASCADE ON UPDATE CASCADE
);
```

**الفهارس**:
- PRIMARY KEY: `id`
- UNIQUE: (`bot_token`, `user_id`) - مستخدم واحد لكل بوت
- INDEX: `bot_token`, `user_id`

**الوظيفة**: تتبع المستخدمين المسموح لهم في كل بوت

**ملاحظات**:
- `banned`: حظر داخل البوت (ليس نظام الحدود)
- عند حذف بوت، يتم حذف جميع مستخدميه تلقائياً

---

### 3. جدول `steam_accounts` (حسابات Steam)

```sql
CREATE TABLE `steam_accounts` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `account_name` varchar(64) UNIQUE NOT NULL,
  `shared_secret` varchar(255) NOT NULL,
  `active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT current_timestamp()
);
```

**الفهارس**:
- PRIMARY KEY: `id`
- UNIQUE: `account_name`

**الوظيفة**: تخزين معلومات الحسابات لتوليد أكواد Steam Guard

**ملاحظات**:
- `shared_secret`: مفتاح TOTP (base64 encoded)
- `active`: تفعيل/تعطيل الحساب

---

### 4. جدول `account_groups` (مجموعات الحسابات)

```sql
CREATE TABLE `account_groups` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `group_name` varchar(64) UNIQUE NOT NULL,
  `description` text DEFAULT NULL,
  `is_default` tinyint(1) DEFAULT 0,
  `active` tinyint(1) DEFAULT 1,
  `locked` tinyint(1) DEFAULT 0,  -- قفل يمنع rename/delete
  `priority` int(11) DEFAULT 0,
  `created_at` timestamp DEFAULT current_timestamp(),
  `group_name_lower` varchar(64) GENERATED ALWAYS AS (lcase(`group_name`)) STORED
);
```

**الفهارس**:
- PRIMARY KEY: `id`
- UNIQUE: `group_name`
- INDEX: `group_name_lower` (للبحث case-insensitive)

**الوظيفة**: تنظيم الحسابات في مجموعات

**أمثلة**:
- default, 36k, 100k, 200k

**ملاحظات**:
- `is_default`: المجموعة العامة
- `locked`: حماية من الحذف/التعديل
- `priority`: ترتيب البحث (غير مستخدم حالياً)

---

### 5. جدول `bot_groups` (ربط البوتات بالمجموعات)

```sql
CREATE TABLE `bot_groups` (
  `bot_id` int(11) NOT NULL,
  `group_id` int(11) NOT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`bot_id`, `group_id`),
  FOREIGN KEY (`bot_id`) REFERENCES `sub_bots`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`group_id`) REFERENCES `account_groups`(`id`) ON DELETE CASCADE
);
```

**الوظيفة**: علاقة Many-to-Many بين البوتات والمجموعات

**مثال**: بوت "test1" مربوط بمجموعة "36k"

---

### 6. جدول `group_accounts` (ربط الحسابات بالمجموعات)

```sql
CREATE TABLE `group_accounts` (
  `group_id` int(11) NOT NULL,
  `account_id` int(11) NOT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`group_id`, `account_id`),
  FOREIGN KEY (`group_id`) REFERENCES `account_groups`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`account_id`) REFERENCES `steam_accounts`(`id`) ON DELETE CASCADE
);
```

**الوظيفة**: علاقة Many-to-Many بين المجموعات والحسابات

**مثال**: الحساب "acc1" موجود في مجموعة "36k"

---

### 7. جدول `limit_counters` (عدادات الاستخدام)

```sql
CREATE TABLE `limit_counters` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `user_id` bigint(20) NOT NULL,
  `account_id` int(11) NOT NULL,
  `bot_label` varchar(64) NOT NULL,
  `day_used` int(11) DEFAULT 0,
  `week_used` int(11) DEFAULT 0,
  `month_used` int(11) DEFAULT 0,
  `day_reset_at` timestamp NULL,
  `week_reset_at` timestamp NULL,
  `month_reset_at` timestamp NULL,
  `updated_at` timestamp DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  UNIQUE KEY (`user_id`, `account_id`, `bot_label`)
);
```

**الفهارس**:
- PRIMARY KEY: `id`
- UNIQUE: (`user_id`, `account_id`, `bot_label`)
- INDEX: `user_id`, `account_id`, `bot_label`

**الوظيفة**: تتبع عدد الأكواد المسحوبة لكل مستخدم في كل حساب في كل بوت

**ملاحظات**:
- **مستقل لكل بوت** (`bot_label`)
- يتم إعادة تعيين العدادات تلقائياً عند انتهاء المدة

---

### 8. جدول `limit_bans` (الحظر من الحسابات)

```sql
CREATE TABLE `limit_bans` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `user_id` bigint(20) NOT NULL,
  `account_id` int(11) NOT NULL,
  `bot_label` varchar(64) NOT NULL,
  `banned_until` timestamp NOT NULL,
  `reason` varchar(255) DEFAULT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  UNIQUE KEY (`user_id`, `account_id`, `bot_label`)
);
```

**الوظيفة**: حظر مؤقت من استخدام حساب معين في بوت معين

**ملاحظات**:
- **مستقل لكل بوت** (`bot_label`)
- يتم حذف الحظر المنتهي عبر `cleanup_expired_bans()`

---

### 9. جدول `limits_global` (الحدود العامة)

```sql
CREATE TABLE `limits_global` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `mode` enum('daily','weekly','monthly') DEFAULT 'daily',
  `per_day` int(11) DEFAULT 2,
  `weekly_cap` int(11) DEFAULT NULL,
  `monthly_cap` int(11) DEFAULT NULL,
  `ban_days` int(11) DEFAULT NULL
);
```

**الوظيفة**: الحدود الافتراضية لجميع البوتات والحسابات

**الأولوية**: 3 (الأخيرة)

---

### 10. جدول `limits_bot` (حدود البوت)

```sql
CREATE TABLE `limits_bot` (
  `bot_label` varchar(64) PRIMARY KEY,
  `mode` enum('daily','weekly','monthly') NOT NULL,
  `per_day` int(11) NOT NULL,
  `weekly_cap` int(11) DEFAULT NULL,
  `monthly_cap` int(11) DEFAULT NULL,
  `ban_days` int(11) DEFAULT NULL
);
```

**الوظيفة**: حدود خاصة ببوت معين (تتجاوز الحدود العامة)

**الأولوية**: 2

---

### 11. جدول `limits_override` (تجاوز الحدود للحسابات)

```sql
CREATE TABLE `limits_override` (
  `account_id` int(11) PRIMARY KEY,
  `mode` enum('daily','weekly','monthly') NOT NULL,
  `per_day` int(11) NOT NULL,
  `cap` int(11) DEFAULT NULL,
  `ban_days` int(11) DEFAULT NULL,
  FOREIGN KEY (`account_id`) REFERENCES `steam_accounts`(`id`) ON DELETE CASCADE
);
```

**الوظيفة**: تجاوز حدود لحساب معين (أعلى أولوية)

**الأولوية**: 1 (الأولى)

---

### 12. جدول `vouchers` (القسائم)

```sql
CREATE TABLE `vouchers` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `code` varchar(32) UNIQUE NOT NULL,
  `voucher_type` enum('users','days','unlimited') DEFAULT 'users',
  `amount` int(11) NOT NULL,
  `target_label` varchar(64) DEFAULT NULL,  -- NULL = للجميع
  `max_uses` int(11) DEFAULT 1,
  `current_uses` int(11) DEFAULT 0,
  `created_by_user_id` bigint(20) NOT NULL,
  `created_at` timestamp DEFAULT current_timestamp()
);
```

**الوظيفة**: تخزين القسائم المتاحة

**أنواع القسائم**:
- `users`: زيادة max_users
- `days`: (غير مفعل حالياً)
- `unlimited`: (غير مفعل حالياً)

---

### 13. جدول `bot_managers` (المدراء الفرعيون)

```sql
CREATE TABLE `bot_managers` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `user_id` bigint(20) NOT NULL,
  `bot_token` varchar(128) DEFAULT NULL,  -- NULL = مدير عام
  `permissions` varchar(255) NOT NULL,    -- '1:2:3' أو 'all'
  `username` varchar(64) DEFAULT NULL,
  `added_by` bigint(20) NOT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT current_timestamp(),
  `updated_at` timestamp DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  UNIQUE KEY (`user_id`, `bot_token`),
  FOREIGN KEY (`bot_token`) REFERENCES `sub_bots`(`bot_token`) ON DELETE CASCADE
);
```

**الصلاحيات**:
1. بث (Broadcast)
2. حظر (Ban)
3. تعبئة قسائم (Redeem)
4. عرض المستخدمين (View Users)
5. البحث (Search)
6. تعديل الحدود (Modify Limits)

---

## 🔍 Views (العروض)

### 1. VIEW: `active_bans`

```sql
CREATE VIEW `active_bans` AS 
SELECT 
  lb.id, lb.user_id, lb.account_id, lb.banned_until, 
  lb.reason, lb.created_at, sa.account_name,
  TIMESTAMPDIFF(HOUR, CURRENT_TIMESTAMP, lb.banned_until) AS hours_remaining
FROM limit_bans lb
JOIN steam_accounts sa ON sa.id = lb.account_id
WHERE lb.banned_until > CURRENT_TIMESTAMP;
```

**الوظيفة**: عرض الحظورات النشطة مع حساب الساعات المتبقية

---

### 2. VIEW: `current_usage`

```sql
CREATE VIEW `current_usage` AS 
SELECT 
  lc.id, lc.user_id, lc.account_id, 
  lc.day_used, lc.week_used, lc.month_used,
  lc.day_reset_at, lc.week_reset_at, lc.month_reset_at,
  lc.updated_at, sa.account_name,
  lc.day_used AS current_day_used,
  lc.week_used AS current_week_used,
  lc.month_used AS current_month_used
FROM limit_counters lc
JOIN steam_accounts sa ON sa.id = lc.account_id;
```

**الوظيفة**: عرض الاستخدام الحالي مع أسماء الحسابات

---

## ⚙️ Stored Procedures

### 1. `cleanup_expired_bans()`

```sql
CREATE PROCEDURE `cleanup_expired_bans`()
BEGIN
  DELETE FROM `limit_bans` WHERE `banned_until` < NOW();
END;
```

**الوظيفة**: حذف الحظر المنتهي تلقائياً

**يتم استدعاؤها**: كل 5 دقائق عبر Cron Job

---

### 2. `reset_expired_counters()`

```sql
CREATE PROCEDURE `reset_expired_counters`()
BEGIN
  -- إعادة تعيين العدادات اليومية
  UPDATE `limit_counters` 
  SET `day_used` = 0,
      `day_reset_at` = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
  WHERE `day_reset_at` < NOW();
  
  -- إعادة تعيين العدادات الأسبوعية
  UPDATE `limit_counters` 
  SET `week_used` = 0,
      `week_reset_at` = DATE_ADD(
        DATE_ADD(CURDATE(), INTERVAL (7 - WEEKDAY(CURDATE())) DAY),
        INTERVAL 1 WEEK
      )
  WHERE `week_reset_at` < NOW();
  
  -- إعادة تعيين العدادات الشهرية
  UPDATE `limit_counters` 
  SET `month_used` = 0,
      `month_reset_at` = DATE_ADD(
        LAST_DAY(CURDATE()),
        INTERVAL 1 DAY
      )
  WHERE `month_reset_at` < NOW();
END;
```

**الوظيفة**: إعادة تعيين العدادات المنتهية

**يتم استدعاؤها**: كل 5 دقائق عبر Cron Job

---

## 🎯 ترتيب أولويات الحدود

```
1️⃣ limits_override (لحساب معين)
       ↓ (إذا لم يوجد)
2️⃣ limits_bot (لبوت معين)
       ↓ (إذا لم يوجد)
3️⃣ limits_global (للجميع)
       ↓ (إذا لم يوجد)
افتراضي: daily, 2 per_day
```

---

## 📊 الفهارس المهمة (Indexes)

### فهارس للأداء:

```sql
-- للبحث السريع عن المستخدمين
INDEX idx_user (allowed_users.user_id)
INDEX idx_bot (allowed_users.bot_token)

-- للبحث في الطلبات
INDEX idx_requests_user (steam_requests.user_id)
INDEX idx_requests_bot (steam_requests.bot_token)
INDEX idx_requests_time (steam_requests.request_time)

-- لنظام الحدود
INDEX idx_counters_user (limit_counters.user_id)
INDEX idx_bans_user (limit_bans.user_id)

-- للمجموعات
INDEX idx_group_accounts (group_accounts.group_id, account_id)
```

---

## 🔐 القيود (Constraints)

### Foreign Keys النشطة:

1. `allowed_users.bot_token` → `sub_bots.bot_token`
2. `bot_groups.bot_id` → `sub_bots.id`
3. `bot_groups.group_id` → `account_groups.id`
4. `group_accounts.group_id` → `account_groups.id`
5. `group_accounts.account_id` → `steam_accounts.id`
6. `limits_override.account_id` → `steam_accounts.id`
7. `bot_managers.bot_token` → `sub_bots.bot_token`
8. `voucher_redemptions.bot_token` → `sub_bots.bot_token`

### UNIQUE Constraints:

- `sub_bots`: label, bot_token
- `steam_accounts`: account_name
- `account_groups`: group_name
- `allowed_users`: (bot_token, user_id)
- `limit_counters`: (user_id, account_id, bot_label)
- `limit_bans`: (user_id, account_id, bot_label)
- `vouchers`: code

---

## 📝 ملاحظات مهمة

### 1. استقلالية العدادات والحظر

العدادات والحظر **مستقلة لكل بوت** عبر حقل `bot_label`:
- مستخدم يمكن أن يستخدم الحساب "acc1" في البوت "bot1" مرتين
- ونفس المستخدم يمكن أن يستخدمه في "bot2" مرتين أيضاً
- العدادات لا تتداخل

### 2. نظام المجموعات

- البوت يرى فقط الحسابات الموجودة في مجموعاته
- استعلام معقد يربط 4 جداول: `steam_accounts` ← `group_accounts` ← `account_groups` ← `bot_groups`

### 3. جداول قديمة (Legacy)

بعض الجداول/الحقول لم تعد تُستخدم:
- `sub_bots.daily_attempts` (استُبدل بنظام الحدود)
- `voucher_usage` (استُبدل بـ `voucher_redemptions`)

---

## 🎯 الخلاصة

قاعدة البيانات مصممة بشكل احترافي مع:
- ✅ Foreign Keys للحفاظ على سلامة البيانات
- ✅ Indexes لتحسين الأداء
- ✅ UNIQUE Constraints لمنع التكرار
- ✅ Views لتسهيل الاستعلامات المعقدة
- ✅ Stored Procedures للصيانة الدورية
