<?php
declare(strict_types=1);

require __DIR__ . '/lib/help_content.php';
require __DIR__ . '/lib/telegram.php';

$cfg = env_cfg();
$OWNER = (int)$cfg['OWNER_USER_ID'];
$CONTROL_TOKEN = (string)$cfg['CONTROL_BOT_TOKEN'];
$DOMAIN = rtrim((string)$cfg['DOMAIN'], '/');

$update = json_decode(file_get_contents('php://input') ?: '[]', true);
if (!$update) { http_response_code(200); exit('OK'); }

$msg = $update['message'] ?? $update['edited_message'] ?? null;
if (!$msg) { http_response_code(200); exit('OK'); }

// حماية من المعالجة المكررة
$update_id = $update['update_id'] ?? 0;
if ($update_id > 0) {
    $cache_file = sys_get_temp_dir() . '/tg_update_' . $update_id;
    if (file_exists($cache_file)) {
        http_response_code(200);
        exit('OK'); // Update already processed
    }
    file_put_contents($cache_file, '1');
    // حذف ملفات قديمة (أكثر من ساعة)
    foreach (glob(sys_get_temp_dir() . '/tg_update_*') as $old_file) {
        if (time() - filemtime($old_file) > 3600) {
            @unlink($old_file);
        }
    }
}

$chat_id = $msg['chat']['id'] ?? 0;
$from_id = $msg['from']['id'] ?? 0;
$text = trim($msg['text'] ?? '');

// Only owner allowed
if ($from_id !== $OWNER) {
    tg_send($CONTROL_TOKEN, $chat_id, "⛔️ ليس لديك صلاحية التحكم بهذا البوت.");
    exit;
}

// Check if owner is in a session (awaiting welcome, etc.)
$db = pdo();
$session = null;
if ($text && strpos($text, '/') !== 0) {
    $stmt = $db->prepare("SELECT * FROM control_sessions WHERE user_id=? ORDER BY id DESC LIMIT 1");
    $stmt->execute([$from_id]);
    $session = $stmt->fetch();
}

if ($session) {
    if ($session['action'] === 'await_welcome' && !empty($session['bot_token'])) {
        $stmt = $db->prepare("UPDATE sub_bots SET welcome_message=? WHERE bot_token=?");
        $stmt->execute([$text, $session['bot_token']]);
        $db->prepare("DELETE FROM control_sessions WHERE id=?")->execute([$session['id']]);
        $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($session['bot_token']);
        $setHook = "https://api.telegram.org/bot{$session['bot_token']}/setWebhook?url=" . urlencode($webhookUrl);
        tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حفظ رسالة الترحيب للبوت.\n\nرابط Webhook:\n{$webhookUrl}\n\n✅ رابط التفعيل الفوري:\n{$setHook}");
        exit;
    }
}

// ============================================
// معالج Callback Queries (للأزرار)
// ============================================
$callback = $update['callback_query'] ?? null;
if ($callback) {
    $from_id = $callback['from']['id'] ?? 0;
    if ($from_id !== $OWNER) {
        http_response_code(200);
        exit('OK');
    }
    
    $data = $callback['data'] ?? '';
    $chat_id = $callback['message']['chat']['id'] ?? 0;
    $msg_id = $callback['message']['message_id'] ?? 0;
    
    tg_call($CONTROL_TOKEN, 'answerCallbackQuery', ['callback_query_id' => $callback['id']]);
    
    if (strpos($data, 'help:') === 0) {
        $section = substr($data, 5);
        $sections = get_help_sections();
        
        if ($section === 'main') {
            tg_call($CONTROL_TOKEN, 'editMessageText', [
                'chat_id' => $chat_id,
                'message_id' => $msg_id,
                'text' => $sections['main']['text'],
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode(['inline_keyboard' => $sections['main']['keyboard']])
            ]);
        } else {
            $pages = get_help_content($section);
            tg_call($CONTROL_TOKEN, 'editMessageText', [
                'chat_id' => $chat_id,
                'message_id' => $msg_id,
                'text' => $pages[0],
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode(['inline_keyboard' => [[['text' => '🔙 القائمة', 'callback_data' => 'help:main']]]])
            ]);
        }
    }
    
    http_response_code(200);
    exit('OK');
}

// Parse commands
if (strpos($text, '/') === 0) {
    $parts = preg_split('/\\s+/', $text);
    $cmd = strtolower($parts[0]);
    $args = array_slice($parts, 1);

    switch ($cmd) {
        case '/start':
            tg_send($CONTROL_TOKEN, $chat_id, "👋 أهلاً يا مالك.\nاستخدم /help لعرض الأوامر.");
            break;

        case '/help': {
            // إرسال القائمة التفاعلية
            $sections = get_help_sections();
            $main = $sections['main'];
            tg_call($CONTROL_TOKEN, 'sendMessage', [
                'chat_id' => $chat_id,
                'text' => $main['text'],
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode(['inline_keyboard' => $main['keyboard']])
            ]);
            
            // إرسال القوائم النصية من قاعدة البيانات
            usleep(500000);
            
            for ($i = 1; $i <= 9; $i++) {
                $part = template_get("help_part{$i}");
                if ($part) {
                    tg_send($CONTROL_TOKEN, $chat_id, $part);
                    usleep(500000);
                }
            }
            
            break;
        }

        case '/addbot': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /addbot <label> <token> [channel] [max_users]\n\n⚠️ ملاحظة: استخدم /help limits لتعيين الحدود");
                break;
            }
            [$label, $token] = $args;
            $channel = $args[2] ?? null;
            $max = isset($args[3]) ? (int)$args[3] : 20;
            $daily = 5; // القيمة الافتراضية (غير مستخدمة - للتوافق فقط)

            // insert
            $stmt = $db->prepare("INSERT INTO sub_bots(label, bot_token, channel_id, max_users, daily_attempts, is_paused) VALUES(?,?,?,?,?,0)");
            try {
                $stmt->execute([$label, $token, $channel, $max, $daily]);
            } catch (Throwable $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل إضافة البوت: " . $e->getMessage());
                break;
            }
            // ask for welcome
            $db->prepare("INSERT INTO control_sessions(user_id, action, bot_token) VALUES(?, 'await_welcome', ?)")
               ->execute([$from_id, $token]);

            $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($token);
            $setHook = "https://api.telegram.org/bot{$token}/setWebhook?url=" . urlencode($webhookUrl);

            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إضافة البوت <b>{$label}</b>.\n\n🔗 Webhook:\n{$webhookUrl}\n\n➡️ أرسل الآن <b>رسالة الترحيب</b> ليتم حفظها.\n\n✅ رابط التفعيل الفوري:\n{$setHook}");
            break;
        }

        case '/list': {
            $bots = $db->query("SELECT b.*, (SELECT COUNT(*) FROM allowed_users u WHERE u.bot_token=b.bot_token) as users_count FROM sub_bots b ORDER BY id DESC")->fetchAll();
            if (!$bots) { tg_send($CONTROL_TOKEN, $chat_id, "لا توجد بوتات."); break; }
            $out = "📋 قائمة البوتات (" . count($bots) . "):\n\n";
            foreach ($bots as $b) {
                $status = $b['is_paused'] ? '⏸️ موقوف' : '▶️ يعمل';
                $out .= "• <b>{$b['label']}</b> — {$status}\n";
                $out .= "  قناة: " . ($b['channel_id'] ?: '—') . " | مستخدمون: {$b['users_count']}\n";
                $out .= "  الحد الأقصى: {$b['max_users']} مستخدم\n";
                $out .= "  token: <code>{$b['bot_token']}</code>\n\n";
            }
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/pause':
        case '/resume': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: {$cmd} <label>"); break; }
            $label = $args[0];
            $val = ($cmd === '/pause') ? 1 : 0;
            $stmt = $db->prepare("UPDATE sub_bots SET is_paused=? WHERE label=?");
            $stmt->execute([$val, $label]);
            tg_send($CONTROL_TOKEN, $chat_id, ($val ? "⏸️ تم إيقاف " : "▶️ تم تشغيل ") . "<b>{$label}</b>");
            break;
        }

        case '/setmax': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /setmax <label> <n>"); break; }
            [$label, $n] = [$args[0], (int)$args[1]];
            $db->prepare("UPDATE sub_bots SET max_users=? WHERE label=?")->execute([$n, $label]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعيين الحد الأقصى للمستخدمين إلى {$n} في <b>{$label}</b>");
            break;
        }

        case '/incmax': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /incmax <label> <delta>"); break; }
            [$label, $d] = [$args[0], (int)$args[1]];
            $db->prepare("UPDATE sub_bots SET max_users = max_users + ? WHERE label=?")->execute([$d, $label]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم زيادة الحد بمقدار {$d} في <b>{$label}</b>");
            break;
        }

        case '/setattempts': {
            tg_send($CONTROL_TOKEN, $chat_id, 
                "⚠️ <b>هذا الأمر معطّل!</b>\n\n" .
                "❌ النظام القديم (daily_attempts) لم يعد يعمل\n\n" .
                "✅ <b>استخدم نظام الحدود الذكي:</b>\n\n" .
                "🌍 <b>حد عام:</b>\n" .
                "<code>/set_daily_all 3</code> - 3 طلبات يومياً للجميع\n\n" .
                "🤖 <b>حد خاص ببوت:</b>\n" .
                "<code>/limit_mode mybot daily</code>\n\n" .
                "👤 <b>حد لحساب محدد:</b>\n" .
                "<code>/override_account 5 daily 10</code>\n\n" .
                "📖 للمزيد: <code>/help limits</code>"
            );
            break;
        }

        case '/setchannel': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /setchannel <label> <channel_id>"); break; }
            [$label, $ch] = [$args[0], $args[1]];
            $db->prepare("UPDATE sub_bots SET channel_id=? WHERE label=?")->execute([$ch, $label]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث قناة الاشتراك في <b>{$label}</b> إلى <code>{$ch}</code>");
            break;
        }

        case '/settoken': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /settoken <label> <new_token>"); break; }
            [$label, $tok] = [$args[0], $args[1]];
            try {
                $db->prepare("UPDATE sub_bots SET bot_token=? WHERE label=?")->execute([$tok, $label]);
            } catch (Throwable $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لم يتم التحديث: " . $e->getMessage());
                break;
            }
            $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($tok);
            $setHook = "https://api.telegram.org/bot{$tok}/setWebhook?url=" . urlencode($webhookUrl);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث توكن <b>{$label}</b>.\n\n🔗 Webhook:\n{$webhookUrl}\n\n✅ رابط التفعيل:\n{$setHook}");
            break;
        }

        case '/setwelcome': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /setwelcome <label> <نص>"); break; }
            $label = array_shift($args);
            $textRest = trim(substr($text, strlen('/setwelcome ' . $label)));
            $db->prepare("UPDATE sub_bots SET welcome_message=? WHERE label=?")->execute([$textRest, $label]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث رسالة الترحيب لـ <b>{$label}</b>.");
            break;
        }

        case '/users': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /users <label>"); break; }
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            if (!$bot) { tg_send($CONTROL_TOKEN, $chat_id, "لم يتم العثور على البوت."); break; }
            $stmt = $db->prepare("SELECT user_id, banned, created_at FROM allowed_users WHERE bot_token=? ORDER BY id DESC LIMIT 200");
            $stmt->execute([$bot['bot_token']]);
            $rows = $stmt->fetchAll();
            if (!$rows) { tg_send($CONTROL_TOKEN, $chat_id, "لا يوجد مستخدمون لهذا البوت."); break; }
            $out = "👥 مستخدمو <b>{$label}</b> (أحدث 200):\n";
            foreach ($rows as $r) {
                $out .= "• {$r['user_id']} " . ($r['banned'] ? '🚫' : '✅') . " — {$r['created_at']}\n";
            }
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/allusers': {
            $rows = $db->query("SELECT DISTINCT user_id FROM allowed_users ORDER BY user_id DESC LIMIT 500")->fetchAll();
            $count = count($rows);
            $out = "👥 إجمالي المستخدمين عبر جميع البوتات: <b>{$count}</b>\n\n";
            
            // تقسيم المستخدمين إلى مجموعات
            $chunks = array_chunk($rows, 100);
            foreach ($chunks as $chunk) {
                $list = implode(", ", array_map(fn($r)=>$r['user_id'], $chunk));
                $out .= $list . "\n";
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, trim($out));
            break;
        }
        
        case '/users_count': {
            $count = $db->query("SELECT COUNT(DISTINCT user_id) as total FROM allowed_users")->fetch()['total'];
            tg_send($CONTROL_TOKEN, $chat_id, "👥 إجمالي المستخدمين: <b>{$count}</b>");
            break;
        }

        case '/ban':
        case '/unban': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: {$cmd} <user_id> [label]"); break; }
            $target = (int)$args[0];
            $label = $args[1] ?? null;
            $val = ($cmd === '/ban') ? 1 : 0;
            if ($label) {
                $stmt = $db->prepare("UPDATE allowed_users u JOIN sub_bots b ON u.bot_token=b.bot_token SET u.banned=? WHERE u.user_id=? AND b.label=?");
                $stmt->execute([$val, $target, $label]);
            } else {
                $stmt = $db->prepare("UPDATE allowed_users SET banned=? WHERE user_id=?");
                $stmt->execute([$val, $target]);
            }
            tg_send($CONTROL_TOKEN, $chat_id, ($val ? "🚫 تم حظر " : "✅ تم فك الحظر عن ") . $target . ($label ? " في {$label}" : " في جميع البوتات"));
            break;
        }

        case '/checkuser': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /checkuser <user_id>"); break; }
            $uid = (int)$args[0];
            $out = "📊 نشاط المستخدم {$uid}:\n";
            $stmt = $db->prepare("SELECT r.bot_token, b.label, COUNT(*) cnt, MAX(r.request_time) last_time
                                  FROM steam_requests r JOIN sub_bots b ON r.bot_token=b.bot_token
                                  WHERE r.user_id=? GROUP BY r.bot_token, b.label ORDER BY last_time DESC");
            $stmt->execute([$uid]);
            $rows = $stmt->fetchAll();
            if (!$rows) { tg_send($CONTROL_TOKEN, $chat_id, "لا يوجد نشاط مسجل لهذا المستخدم."); break; }
            foreach ($rows as $r) {
                $out .= "• {$r['label']}: {$r['cnt']} طلب (آخر: {$r['last_time']})\n";
            }
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/accounts': {
            $rows = $db->query("SELECT account_name, active, daily_attempts_override FROM steam_accounts ORDER BY account_name ASC")->fetchAll();
            if (!$rows) { tg_send($CONTROL_TOKEN, $chat_id, "لا توجد حسابات."); break; }
            $out = "🎮 الحسابات:\n";
            foreach ($rows as $r) {
                $out .= "• {$r['account_name']} — " . ($r['active'] ? '✅' : '⛔️') . " ";
                if ($r['daily_attempts_override'] !== null) $out .= "(محاولات/يوم: {$r['daily_attempts_override']})";
                $out .= "\n";
            }
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/addaccount': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /addaccount <name> <shared_secret_b64>"); break; }
            [$name, $sec] = [$args[0], $args[1]];
            $stmt = $db->prepare("INSERT INTO steam_accounts(account_name, shared_secret) VALUES(?,?) ON DUPLICATE KEY UPDATE shared_secret=VALUES(shared_secret), active=1");
            $stmt->execute([$name, $sec]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تمت إضافة/تحديث الحساب <b>{$name}</b>.");
            break;
        }

        case '/addaccounts': {
            // Accept multi-line format in the SAME message: each line "name:secret"
            $payload = trim(substr($text, strlen('/addaccounts')));
            if ($payload === '') {
                tg_send($CONTROL_TOKEN, $chat_id, "أرسل بنفس الرسالة بعد الأمر قائمة بصيغة:\nname1:secret1\nname2:secret2");
                break;
            }
            $lines = preg_split('/\\r?\\n/', $payload);
            $ok = 0; $fail = 0;
            foreach ($lines as $ln) {
                $ln = trim($ln);
                if ($ln === '') continue;
                $pos = strpos($ln, ':');
                if ($pos === false) { $fail++; continue; }
                $name = trim(substr($ln, 0, $pos));
                $sec = trim(substr($ln, $pos+1));
                if ($name === '' || $sec === '') { $fail++; continue; }
                $stmt = $db->prepare("INSERT INTO steam_accounts(account_name, shared_secret) VALUES(?,?) ON DUPLICATE KEY UPDATE shared_secret=VALUES(shared_secret), active=1");
                try { $stmt->execute([$name, $sec]); $ok++; } catch (Throwable $e) { $fail++; }
            }
            tg_send($CONTROL_TOKEN, $chat_id, "تمت معالجة الدفعة: ✅ ناجحة {$ok} | ❌ فاشلة {$fail}");
            break;
        }

        case '/editaccount': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /editaccount <name> <shared_secret_b64|same> [active=0|1]"); break; }
            $name = $args[0];
            $sec = $args[1];
            $active = isset($args[2]) ? (int)$args[2] : null;
            if ($sec !== 'same') {
                $db->prepare("UPDATE steam_accounts SET shared_secret=? WHERE account_name=?")->execute([$sec, $name]);
            }
            if ($active !== null) {
                $db->prepare("UPDATE steam_accounts SET active=? WHERE account_name=?")->execute([$active, $name]);
            }
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعديل الحساب <b>{$name}</b>.");
            break;
        }

        case '/disableaccount':
        case '/enableaccount': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: {$cmd} <name>"); break; }
            $name = $args[0];
            $val = ($cmd === '/enableaccount') ? 1 : 0;
            $db->prepare("UPDATE steam_accounts SET active=? WHERE account_name=?")->execute([$val, $name]);
            tg_send($CONTROL_TOKEN, $chat_id, ($val ? "✅ تم تفعيل" : "⛔️ تم تعطيل") . " الحساب <b>{$name}</b>.");
            break;
        }

        case '/responses': {
            try {
                $rows = $db->query("SELECT `key`,`text` FROM message_templates ORDER BY `key` ASC")->fetchAll();
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ في قاعدة البيانات:\n" . $e->getMessage());
                break;
            }
            
            if (!$rows || count($rows) == 0) { 
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد ردود عامة مخزنة.\n\n💡 تأكد من تطبيق SQL:\n<code>mysql -u user -p db < sql/supehgku_test0.sql</code>"); 
                break; 
            }
            
            $out = "🗂 <b>الردود العامة (Global Templates):</b>\n\n";
            $out .= "📊 <b>العدد:</b> " . count($rows) . "\n\n";
            
            foreach ($rows as $r) {
                $preview = mb_substr($r['text'], 0, 80);
                if (mb_strlen($r['text']) > 80) {
                    $preview .= '...';
                }
                // هروب HTML entities لتجنب التضارب
                $preview_escaped = htmlspecialchars($preview, ENT_QUOTES, 'UTF-8');
                
                $out .= "🔹 <code>{$r['key']}</code>\n";
                $out .= "   {$preview_escaped}\n\n";
                
                // تقسيم الرسالة إذا كانت طويلة
                if (mb_strlen($out) > 3000) {
                    tg_send($CONTROL_TOKEN, $chat_id, $out);
                    $out = "";
                }
            }
            
            // إضافة النصائح دائماً
            if (strlen($out) > 0) {
                $out .= "\n\n💡 <i>لتعديل رد عام:</i> <code>/edit_global_response &lt;key&gt; &lt;text&gt;</code>";
                $out .= "\n💡 <i>لتعديل رد بوت:</i> <code>/edit_bot_response &lt;bot&gt; &lt;key&gt; &lt;text&gt;</code>";
                
                // محاولة الإرسال مع التحقق من النتيجة
                $result = tg_call($CONTROL_TOKEN, 'sendMessage', [
                    'chat_id' => $chat_id,
                    'text' => $out,
                    'parse_mode' => 'HTML',
                    'disable_web_page_preview' => true
                ]);
                
                // إذا فشل، حاول بدون HTML
                if (empty($result['ok'])) {
                    $error_msg = $result['description'] ?? 'Unknown error';
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل الإرسال: {$error_msg}\n\nجاري المحاولة بدون HTML...");
                    
                    tg_call($CONTROL_TOKEN, 'sendMessage', [
                        'chat_id' => $chat_id,
                        'text' => strip_tags($out),
                        'disable_web_page_preview' => true
                    ]);
                }
            } else {
                // إذا تم إرسال كل شيء في الـ loop، أرسل رسالة نصائح فقط
                tg_send($CONTROL_TOKEN, $chat_id, "💡 <i>لتعديل رد عام:</i> <code>/edit_global_response &lt;key&gt; &lt;text&gt;</code>\n💡 <i>لتعديل رد بوت:</i> <code>/edit_bot_response &lt;bot&gt; &lt;key&gt; &lt;text&gt;</code>");
            }
            break;
        }

        case '/editresponse': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /editresponse <key> <new text>"); break; }
            $key = array_shift($args);
            $newText = trim(substr($text, strlen('/editresponse ' . $key)));
            template_set($key, $newText);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث الرد <b>{$key}</b>.");
            break;
        }

        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        // نظام الردود المتقدم - NEW
        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

        case '/edit_global_response': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/edit_global_response &lt;key&gt; &lt;text&gt;</code>\n\n" .
                    "💡 <i>يُحدّث الرد العام في message_templates</i>\n" .
                    "💡 <i>جميع البوتات ستستخدمه ما لم يكن لديها رد مخصص</i>");
                break;
            }
            
            $key = $args[0];
            $newText = trim(substr($text, strlen('/edit_global_response ' . $key)));
            
            if (empty($newText)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ النص فارغ. الرجاء كتابة النص بعد المفتاح.");
                break;
            }
            
            // Check if key exists
            $stmt = $db->prepare("SELECT COUNT(*) as c FROM message_templates WHERE `key`=?");
            $stmt->execute([$key]);
            $exists = $stmt->fetch()['c'] > 0;
            
            if ($exists) {
                $db->prepare("UPDATE message_templates SET text=? WHERE `key`=?")
                   ->execute([$newText, $key]);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث الرد العام <code>{$key}</code>\n\n" .
                    "🌍 سيُطبق على جميع البوتات (ما لم يكن لديها رد مخصص)");
            } else {
                $db->prepare("INSERT INTO message_templates(`key`, text) VALUES(?, ?)")
                   ->execute([$key, $newText]);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إنشاء رد عام جديد <code>{$key}</code>");
            }
            
            // Log
            $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('edit_global_response', ?, ?)")
               ->execute([$key, $from_id]);
            break;
        }

        case '/edit_bot_response': {
            if (count($args) < 3) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/edit_bot_response &lt;bot&gt; &lt;key&gt; &lt;text&gt;</code>\n\n" .
                    "💡 <i>يُنشئ رد مخصص لبوت واحد فقط</i>\n" .
                    "💡 <i>يتجاوز الرد العام</i>\n\n" .
                    "<b>مثال:</b>\n<code>/edit_bot_response mybot welcome_msg مرحباً في بوتي الخاص!</code>");
                break;
            }
            
            $bot_label = $args[0];
            $key = $args[1];
            $newText = trim(substr($text, strlen("/edit_bot_response {$bot_label} {$key} ")));
            
            if (empty($newText)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ النص فارغ. الرجاء كتابة النص بعد المفتاح.");
                break;
            }
            
            // Get bot
            $stmt = $db->prepare("SELECT id, label FROM sub_bots WHERE LOWER(label)=LOWER(?)");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // Insert or update
            $db->prepare("INSERT INTO bot_message_templates(bot_id, template_key, template_text) 
                         VALUES(?, ?, ?)
                         ON DUPLICATE KEY UPDATE template_text=VALUES(template_text)")
               ->execute([$bot['id'], $key, $newText]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث رد البوت <b>{$bot['label']}</b>\n\n" .
                "🔑 المفتاح: <code>{$key}</code>\n" .
                "🤖 البوت: <b>{$bot['label']}</b>\n" .
                "💬 النص: {$newText}");
            
            // Log
            $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('edit_bot_response', ?, ?)")
               ->execute(["{$bot['label']}: {$key}", $from_id]);
            break;
        }

        case '/bot_responses': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/bot_responses &lt;bot&gt;</code>\n\n" .
                    "💡 <i>يعرض الردود المخصصة للبوت</i>");
                break;
            }
            
            $bot_label = $args[0];
            
            // Get bot
            $stmt = $db->prepare("SELECT id, label FROM sub_bots WHERE LOWER(label)=LOWER(?)");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // Get custom responses
            $stmt = $db->prepare("SELECT template_key, template_text, updated_at 
                                 FROM bot_message_templates 
                                 WHERE bot_id=? 
                                 ORDER BY template_key");
            $stmt->execute([$bot['id']]);
            $responses = $stmt->fetchAll();
            
            if (empty($responses)) {
                tg_send($CONTROL_TOKEN, $chat_id, "🤖 <b>البوت: {$bot['label']}</b>\n\n" .
                    "ℹ️ لا توجد ردود مخصصة لهذا البوت\n" .
                    "💡 يستخدم الردود العامة من message_templates\n\n" .
                    "<b>لإنشاء رد مخصص:</b>\n" .
                    "<code>/edit_bot_response {$bot['label']} &lt;key&gt; &lt;text&gt;</code>");
                break;
            }
            
            $out = "🤖 <b>ردود البوت: {$bot['label']}</b>\n\n";
            $out .= "📊 <b>عدد الردود المخصصة:</b> " . count($responses) . "\n\n";
            
            foreach ($responses as $r) {
                $preview = mb_substr($r['template_text'], 0, 60);
                if (mb_strlen($r['template_text']) > 60) {
                    $preview .= '...';
                }
                $out .= "🔹 <code>{$r['template_key']}</code>\n";
                $out .= "   {$preview}\n";
                $out .= "   <i>آخر تحديث: {$r['updated_at']}</i>\n\n";
            }
            
            $out .= "\n💡 <i>لتعديل:</i> <code>/edit_bot_response {$bot['label']} &lt;key&gt; &lt;text&gt;</code>";
            $out .= "\n💡 <i>لحذف:</i> <code>/delete_bot_response {$bot['label']} &lt;key&gt;</code>";
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/delete_bot_response': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/delete_bot_response &lt;bot&gt; &lt;key&gt;</code>");
                break;
            }
            
            $bot_label = $args[0];
            $key = $args[1];
            
            // Get bot
            $stmt = $db->prepare("SELECT id, label FROM sub_bots WHERE LOWER(label)=LOWER(?)");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // Delete
            $stmt = $db->prepare("DELETE FROM bot_message_templates WHERE bot_id=? AND template_key=?");
            $stmt->execute([$bot['id'], $key]);
            
            if ($stmt->rowCount() > 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف الرد المخصص <code>{$key}</code> من البوت <b>{$bot['label']}</b>\n\n" .
                    "💡 سيستخدم البوت الآن الرد العام إن وُجد");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لم يتم العثور على رد <code>{$key}</code> للبوت <b>{$bot['label']}</b>");
            }
            break;
        }

        case '/copy_global_to_bot': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/copy_global_to_bot &lt;key&gt; &lt;bot&gt;</code>\n\n" .
                    "💡 <i>ينسخ رد عام ليصبح رد مخصص لبوت</i>");
                break;
            }
            
            $key = $args[0];
            $bot_label = $args[1];
            
            // Get global template
            $stmt = $db->prepare("SELECT text FROM message_templates WHERE `key`=?");
            $stmt->execute([$key]);
            $global = $stmt->fetch();
            
            if (!$global) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الرد العام <code>{$key}</code> غير موجود");
                break;
            }
            
            // Get bot
            $stmt = $db->prepare("SELECT id, label FROM sub_bots WHERE LOWER(label)=LOWER(?)");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // Copy
            $db->prepare("INSERT INTO bot_message_templates(bot_id, template_key, template_text) 
                         VALUES(?, ?, ?)
                         ON DUPLICATE KEY UPDATE template_text=VALUES(template_text)")
               ->execute([$bot['id'], $key, $global['text']]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم نسخ الرد العام <code>{$key}</code> إلى البوت <b>{$bot['label']}</b>\n\n" .
                "💡 يمكنك الآن تعديله بـ:\n<code>/edit_bot_response {$bot['label']} {$key} &lt;text&gt;</code>");
            break;
        }

        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

        case '/broadcast': {
            if (count($args) < 2) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /broadcast <label> <text>"); break; }
            $label = array_shift($args);
            $payload = trim(substr($text, strlen('/broadcast ' . $label)));
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            if (!$bot) { tg_send($CONTROL_TOKEN, $chat_id, "لم يتم العثور على البوت."); break; }
            $tok = $bot['bot_token'];
            $stmt = $db->prepare("SELECT user_id FROM allowed_users WHERE bot_token=? AND banned=0");
            $stmt->execute([$tok]);
            $users = $stmt->fetchAll();
            $sent=0; $fail=0;
            foreach ($users as $u) {
                $r = tg_call($tok, 'sendMessage', ['chat_id'=>$u['user_id'], 'text'=>$payload, 'parse_mode'=>'HTML', 'disable_web_page_preview'=>true]);
                if (!empty($r['ok'])) $sent++; else $fail++;
                usleep(200000); // 0.2s to avoid hitting limits
            }
            tg_send($CONTROL_TOKEN, $chat_id, "📢 تم الإرسال إلى مستخدمي <b>{$label}</b>: ✅ {$sent} | ❌ {$fail}");
            break;
        }

        case '/broadcast_all': {
            if (count($args) < 1) { tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /broadcast_all <text>"); break; }
            $payload = trim(substr($text, strlen('/broadcast_all')));
            $bots = $db->query("SELECT bot_token,label FROM sub_bots")->fetchAll();
            $sent=0; $fail=0;
            foreach ($bots as $b) {
                $stmt = $db->prepare("SELECT user_id FROM allowed_users WHERE bot_token=? AND banned=0");
                $stmt->execute([$b['bot_token']]);
                $users = $stmt->fetchAll();
                foreach ($users as $u) {
                    $r = tg_call($b['bot_token'], 'sendMessage', ['chat_id'=>$u['user_id'], 'text'=>$payload, 'parse_mode'=>'HTML', 'disable_web_page_preview'=>true]);
                    if (!empty($r['ok'])) $sent++; else $fail++;
                    usleep(200000);
                }
            }
            tg_send($CONTROL_TOKEN, $chat_id, "📣 تم الإرسال للجميع: ✅ {$sent} | ❌ {$fail}");
            break;
        }

        // ============================================
        // الأوامر الجديدة v2.0
        // ============================================

        case '/deletebot': {
            if (count($args) < 2 || $args[1] !== 'confirm') {
                tg_send($CONTROL_TOKEN, $chat_id, "⚠️ لحذف البوت نهائياً، أرسل:\n<code>/deletebot {$args[0]} confirm</code>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت غير موجود");
                break;
            }
            
            $db->prepare("DELETE FROM sub_bots WHERE label=?")->execute([$label]);
            
            $deleteHook = "https://api.telegram.org/bot{$bot['bot_token']}/deleteWebhook";
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف البوت <b>{$label}</b> نهائياً\n\n🔗 لحذف Webhook:\n{$deleteHook}");
            break;
        }

        case '/refreshbot': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /refreshbot <label>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت غير موجود");
                break;
            }
            
            $tok = $bot['bot_token'];
            $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($tok);
            
            $result = tg_call($tok, 'setWebhook', ['url' => $webhookUrl]);
            
            if (!empty($result['ok'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث Webhook لـ <b>{$label}</b>\n\n🔗 {$webhookUrl}");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل التحديث: " . ($result['description'] ?? 'خطأ'));
            }
            break;
        }

        case '/deleteaccount': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /deleteaccount <name>");
                break;
            }
            
            $name = $args[0];
            $db->prepare("DELETE FROM steam_accounts WHERE account_name=?")->execute([$name]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف الحساب <b>{$name}</b> نهائياً");
            break;
        }

        case '/accountinfo': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /accountinfo <name>");
                break;
            }
            
            $name = $args[0];
            $stmt = $db->prepare("SELECT * FROM steam_accounts WHERE account_name=?");
            $stmt->execute([$name]);
            $acc = $stmt->fetch();
            
            if (!$acc) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحساب غير موجود");
                break;
            }
            
            $stmt = $db->prepare("SELECT COUNT(*) c, MAX(request_time) last FROM steam_requests WHERE account_name=?");
            $stmt->execute([$name]);
            $req = $stmt->fetch();
            
            $out = "🎮 <b>معلومات الحساب: {$name}</b>\n\n";
            $out .= "الحالة: " . ($acc['active'] ? '✅ نشط' : '⛔️ معطل') . "\n";
            $out .= "إجمالي الطلبات: " . ($req['c'] ?? 0) . "\n";
            $out .= "آخر استخدام: " . ($req['last'] ?? '—') . "\n";
            $out .= "تاريخ الإنشاء: {$acc['created_at']}\n";
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/reset_limits': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /reset_limits <user_id>");
                break;
            }
            
            $uid = (int)$args[0];
            $since = date('Y-m-d H:i:s', time() - 86400);
            
            $stmt = $db->prepare("DELETE FROM steam_requests WHERE user_id=? AND request_time >= ?");
            $stmt->execute([$uid, $since]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إعادة ضبط حدود المستخدم {$uid}");
            break;
        }

        case '/system_status': {
            $status = "🔧 <b>حالة النظام</b>\n\n";
            
            try {
                $db->query("SELECT 1");
                $status .= "✅ قاعدة البيانات: متصلة\n";
            } catch(Exception $e) {
                $status .= "❌ قاعدة البيانات: خطأ\n";
            }
            
            $bots = $db->query("SELECT COUNT(*) c FROM sub_bots")->fetch()['c'];
            $users = $db->query("SELECT COUNT(*) c FROM allowed_users")->fetch()['c'];
            $accounts = $db->query("SELECT COUNT(*) c FROM steam_accounts WHERE active=1")->fetch()['c'];
            
            $status .= "📊 البوتات: {$bots}\n";
            $status .= "👥 المستخدمون: {$users}\n";
            $status .= "🎮 الحسابات النشطة: {$accounts}\n";
            
            // فحص المجموعات إن وجدت
            try {
                $groups = $db->query("SELECT COUNT(*) c FROM account_groups")->fetch()['c'];
                $status .= "📂 المجموعات: {$groups}\n";
            } catch(Exception $e) {}
            
            tg_send($CONTROL_TOKEN, $chat_id, $status);
            break;
        }

        // ============================================
        // أوامر القسائم v2.0
        // ============================================

        case '/generate_voucher': {
            if (count($args) < 2) {
                $usage_msg = "❌ <b>الاستخدام الصحيح:</b>\n\n";
                $usage_msg .= "<code>/generate_voucher &lt;type&gt; &lt;amount&gt; [target] [max_uses]</code>\n\n";
                $usage_msg .= "📦 <b>الأنواع المتاحة:</b>\n";
                $usage_msg .= "• <code>users</code> - مستخدمين\n";
                $usage_msg .= "• <code>days</code> - أيام\n\n";
                $usage_msg .= "🎯 <b>الهدف (target):</b>\n";
                $usage_msg .= "• <code>all</code> - جميع البوتات (افتراضي)\n";
                $usage_msg .= "• <code>bot_label</code> - بوت معين\n";
                $usage_msg .= "• <code>group:group_name</code> - مجموعة معينة 🆕\n\n";
                $usage_msg .= "💡 <b>أمثلة:</b>\n";
                $usage_msg .= "• <code>/generate_voucher users 100</code>\n";
                $usage_msg .= "• <code>/generate_voucher users 50 mybot</code>\n";
                $usage_msg .= "• <code>/generate_voucher users 200 group:36k 5</code>\n";
                $usage_msg .= "• <code>/generate_voucher users 300 all 10</code>";
                tg_send($CONTROL_TOKEN, $chat_id, $usage_msg);
                break;
            }
            
            $type = strtolower(trim($args[0]));
            $amount = (int)$args[1];
            $target = $args[2] ?? 'all';
            $max_uses = isset($args[3]) ? (int)$args[3] : 1;
            
            if (!in_array($type, ['users', 'days'])) {
                $error_msg = "❌ <b>نوع خاطئ!</b>\n\n";
                $error_msg .= "📦 الأنواع المتاحة:\n";
                $error_msg .= "• <code>users</code> - مستخدمين\n";
                $error_msg .= "• <code>days</code> - أيام\n\n";
                $error_msg .= "💡 مثال:\n<code>/generate_voucher users {$amount}</code>";
                tg_send($CONTROL_TOKEN, $chat_id, $error_msg);
                break;
            }
            
            if ($amount <= 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ القيمة يجب أن تكون أكبر من صفر!");
                break;
            }
            
            if ($max_uses <= 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ عدد الاستخدامات يجب أن تكون أكبر من صفر!");
                break;
            }
            
            // تحديد نوع الهدف: all, bot, أو group
            $target_label = null;
            $target_group_id = null;
            $target_display = 'جميع البوتات';
            
            if ($target === 'all') {
                // للجميع
                $target_label = null;
                $target_group_id = null;
                $target_display = 'جميع البوتات';
            } elseif (strpos($target, 'group:') === 0) {
                // قسيمة لمجموعة معينة
                $group_name = substr($target, 6);
                $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
                $stmt->execute([$group_name]);
                $group = $stmt->fetch();
                
                if (!$group) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                    break;
                }
                
                $target_group_id = $group['id'];
                $target_display = "مجموعة: <b>{$group_name}</b>";
            } else {
                // قسيمة لبوت معين - التحقق من وجوده
                $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
                $stmt->execute([$target]);
                if (!$stmt->fetch()) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$target}</b> غير موجود");
                    break;
                }
                $target_label = $target;
                $target_display = "بوت: <b>{$target}</b>";
            }
            
            $code_gen = 'V' . strtoupper(bin2hex(random_bytes(5)));
            
            try {
                // التحقق من وجود حقل target_group_id
                $columns = $db->query("SHOW COLUMNS FROM vouchers LIKE 'target_group_id'")->fetchAll();
                $has_group_support = !empty($columns);
                
                if ($target_group_id !== null && !$has_group_support) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ دعم المجموعات غير مفعّل! قم بتنفيذ:\n<code>sql_updates/01_vouchers_groups_support.sql</code>");
                    break;
                }
                
                if ($has_group_support) {
                    $stmt = $db->prepare("INSERT INTO vouchers(code, voucher_type, amount, target_label, target_group_id, max_uses, created_by_user_id) VALUES(?,?,?,?,?,?,?)");
                    $result = $stmt->execute([$code_gen, $type, $amount, $target_label, $target_group_id, $max_uses, $from_id]);
                } else {
                    $stmt = $db->prepare("INSERT INTO vouchers(code, voucher_type, amount, target_label, max_uses, created_by_user_id) VALUES(?,?,?,?,?,?)");
                    $result = $stmt->execute([$code_gen, $type, $amount, $target_label, $max_uses, $from_id]);
                }
                
                if (!$result) {
                    $error_info = $stmt->errorInfo();
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل الإدراج: " . ($error_info[2] ?? 'خطأ غير معروف'));
                    break;
                }
                
                $out = "✅ <b>تم إنشاء القسيمة</b>\n\n";
                $out .= "🎫 الكود: <code>{$code_gen}</code>\n";
                $out .= "📦 النوع: " . ($type === 'users' ? '👥 مستخدمين' : '📅 أيام') . "\n";
                $out .= "💰 القيمة: {$amount}\n";
                $out .= "🎯 الهدف: {$target_display}\n";
                $out .= "🔢 الاستخدامات: {$max_uses}\n\n";
                $out .= "💡 للتفعيل: <code>/redeem {$code_gen}</code>";
                
                tg_send($CONTROL_TOKEN, $chat_id, $out);
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ في إنشاء القسيمة:\n" . $e->getMessage());
            }
            break;
        }

        case '/redeem': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /redeem <code> [bot_label]");
                break;
            }
            
            $code_input = strtoupper($args[0]);
            $stmt = $db->prepare("SELECT * FROM vouchers WHERE code=?");
            $stmt->execute([$code_input]);
            $voucher = $stmt->fetch();
            
            if (!$voucher) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ القسيمة غير موجودة");
                break;
            }
            
            if ($voucher['current_uses'] >= $voucher['max_uses']) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ تم استهلاك القسيمة ({$voucher['current_uses']}/{$voucher['max_uses']})");
                break;
            }
            
            // تحديد البوت المستهدف
            $label = null;
            if ($voucher['target_label']) {
                $label = $voucher['target_label'];
            } elseif (!empty($voucher['target_group_id'])) {
                // قسيمة لمجموعة - يجب تحديد البوت يدوياً
                if (!isset($args[1])) {
                    tg_send($CONTROL_TOKEN, $chat_id, "⚠️ هذه القسيمة للمجموعة، حدد البوت:\n<code>/redeem {$code_input} bot_label</code>");
                    break;
                }
                $label = $args[1];
            } else {
                // قسيمة عامة - يجب تحديد البوت
                if (!isset($args[1])) {
                    tg_send($CONTROL_TOKEN, $chat_id, "⚠️ حدد البوت:\n<code>/redeem {$code_input} bot_label</code>");
                    break;
                }
                $label = $args[1];
            }
            
            // التحقق من وجود البوت
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
                break;
            }
            
            // التحقق من أن البوت ينتمي للمجموعة (إذا كانت القسيمة لمجموعة)
            if (!empty($voucher['target_group_id'])) {
                $stmt = $db->prepare("SELECT COUNT(*) as cnt FROM bot_groups WHERE bot_id=? AND group_id=?");
                $stmt->execute([$bot['id'], $voucher['target_group_id']]);
                $check = $stmt->fetch();
                
                if ($check['cnt'] == 0) {
                    // جلب اسم المجموعة للتوضيح
                    $grp_stmt = $db->prepare("SELECT group_name FROM account_groups WHERE id=?");
                    $grp_stmt->execute([$voucher['target_group_id']]);
                    $grp = $grp_stmt->fetch();
                    $grp_name = $grp['group_name'] ?? 'غير معروف';
                    
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير مرتبط بالمجموعة <b>{$grp_name}</b>\n\n💡 يمكن تفعيل القسيمة فقط في البوتات المرتبطة بهذه المجموعة");
                    break;
                }
            }
            
            // تطبيق القسيمة
            if ($voucher['voucher_type'] === 'users') {
                $db->prepare("UPDATE sub_bots SET max_users = max_users + ? WHERE label=?")->execute([$voucher['amount'], $label]);
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "⚠️ نوع 'days' غير مدعوم حالياً");
                break;
            }
            
            $db->prepare("UPDATE vouchers SET current_uses = current_uses + 1 WHERE code=?")->execute([$code_input]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تم تفعيل القسيمة</b>\n\n🤖 البوت: <b>{$label}</b>\n💰 تمت إضافة: {$voucher['amount']} مستخدم");
            break;
        }

        case '/vouchers': {
    $vouchers = $db->query("SELECT * FROM vouchers ORDER BY id DESC LIMIT 50")->fetchAll();
    
    if (!$vouchers) {
        tg_send($CONTROL_TOKEN, $chat_id, "لا توجد قسائم");
        break;
    }
    
    $out = "🎫 <b>القسائم (آخر 50):</b>\n\n";
    foreach ($vouchers as $v) {
        $status = ($v['current_uses'] >= $v['max_uses']) ? '❌' : '✅';
        $target = $v['target_label'] ? $v['target_label'] : 'الكل';
        $out .= "{$status} <code>{$v['code']}</code>\n";
        $out .= "  └ {$v['voucher_type']} | {$v['amount']} | {$target} | {$v['current_uses']}/{$v['max_uses']}\n\n";
    }
    tg_send($CONTROL_TOKEN, $chat_id, $out);
    break;
}

case '/voucher_search': {
    if (count($args) < 1) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /voucher_search <code>");
        break;
    }
    
    $code_search = strtoupper($args[0]);
    $stmt = $db->prepare("SELECT * FROM vouchers WHERE code=?");
    $stmt->execute([$code_search]);
    $voucher = $stmt->fetch();
    
    if (!$voucher) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ القسيمة غير موجودة");
        break;
    }
    
    $out = "🎫 <b>تفاصيل القسيمة</b>\n\n";
    $out .= "📟 الكود: <code>{$voucher['code']}</code>\n";
    $out .= "📦 النوع: {$voucher['voucher_type']}\n";
    $out .= "💰 القيمة: {$voucher['amount']}\n";
    $out .= "🎯 الهدف: " . ($voucher['target_label'] ?: 'الكل') . "\n";
    $out .= "🔢 الاستخدامات: {$voucher['current_uses']}/{$voucher['max_uses']}\n";
    $out .= "الحالة: " . ($voucher['current_uses'] >= $voucher['max_uses'] ? '❌ مستخدمة' : '✅ نشطة');
    
    tg_send($CONTROL_TOKEN, $chat_id, $out);
    break;
}

// ============================================
// قسم المدراء الفرعيين (Managers)
// ============================================

case '/managers_all': {
    $managers = $db->query("
        SELECT m.*, b.label 
        FROM bot_managers m 
        LEFT JOIN sub_bots b ON m.bot_token = b.bot_token 
        WHERE m.is_active=1 
        ORDER BY m.id DESC
    ")->fetchAll();
    
    if (!$managers) {
        tg_send($CONTROL_TOKEN, $chat_id, "لا يوجد مدراء فرعيون");
        break;
    }
    
    $out = "👥 <b>المدراء الفرعيون:</b>\n\n";
    foreach ($managers as $m) {
        $bot_name = $m['label'] ?: '🌐 جميع البوتات';
        $perms = strtolower(trim($m['permissions'])) === 'all' ? 'كل الصلاحيات' : $m['permissions'];
        $username = $m['username'] ? "@{$m['username']}" : '';
        
        $out .= "👤 <b>ID:</b> <code>{$m['user_id']}</code> {$username}\n";
        $out .= "🤖 <b>البوت:</b> {$bot_name}\n";
        $out .= "🔐 <b>الصلاحيات:</b> {$perms}\n";
        $out .= "📅 <b>أضيف:</b> {$m['created_at']}\n\n";
    }
    
    tg_send($CONTROL_TOKEN, $chat_id, $out);
    break;
}

case '/managers': {
    if (count($args) < 1) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /managers <label>");
        break;
    }
    
    $label = $args[0];
    
    // الحصول على توكن البوت
    $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
    $stmt->execute([$label]);
    $bot = $stmt->fetch();
    
    if (!$bot) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
        break;
    }
    
    $managers = $db->prepare("SELECT * FROM bot_managers WHERE (bot_token=? OR bot_token IS NULL) AND is_active=1 ORDER BY id DESC");
    $managers->execute([$bot['bot_token']]);
    $managers_list = $managers->fetchAll();
    
    if (!$managers_list) {
        tg_send($CONTROL_TOKEN, $chat_id, "لا يوجد مدراء لبوت <b>{$label}</b>");
        break;
    }
    
    $out = "👥 <b>مدراء بوت {$label}:</b>\n\n";
    foreach ($managers_list as $m) {
        $scope = $m['bot_token'] ? "محدد لهذا البوت" : "🌐 مدير عام";
        $perms = strtolower(trim($m['permissions'])) === 'all' ? 'كل الصلاحيات' : $m['permissions'];
        $username = $m['username'] ? "@{$m['username']}" : '';
        
        $out .= "👤 <code>{$m['user_id']}</code> {$username}\n";
        $out .= "📌 {$scope}\n";
        $out .= "🔐 {$perms}\n\n";
    }
    
    tg_send($CONTROL_TOKEN, $chat_id, $out);
    break;
}

case '/addmanager': {
    if (count($args) < 2) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /addmanager <label> <user_id> [permissions]\n\n💡 مثال:\n<code>/addmanager mybot 123456 1:2:3</code>\n<code>/addmanager mybot 123456 all</code>\n\n📝 الصلاحيات:\n1️⃣ بث\n2️⃣ حظر\n3️⃣ تعبئة قسائم\n4️⃣ عرض المستخدمين\n5️⃣ البحث\n6️⃣ تعديل الحدود");
        break;
    }
    
    $label = $args[0];
    $manager_id = (int)$args[1];
    $perms = $args[2] ?? '1:2:3:4:5';
    
    // التحقق من البوت
    $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
    $stmt->execute([$label]);
    $bot = $stmt->fetch();
    
    if (!$bot) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
        break;
    }
    
    // إضافة المدير
    $success = add_manager($manager_id, $bot['bot_token'], $perms, $from_id);
    
    if ($success) {
        tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تمت إضافة المدير</b>\n\n👤 ID: <code>{$manager_id}</code>\n🤖 البوت: <b>{$label}</b>\n🔐 الصلاحيات: {$perms}");
    } else {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل في إضافة المدير");
    }
    break;
}

case '/addmanager_all': {
    if (count($args) < 1) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /addmanager_all <user_id> [permissions]\n\n💡 مثال:\n<code>/addmanager_all 123456 all</code>\n<code>/addmanager_all 123456 1:2:3</code>");
        break;
    }
    
    $manager_id = (int)$args[0];
    $perms = $args[1] ?? 'all';
    
    // إضافة مدير عام (لكل البوتات)
    $success = add_manager($manager_id, null, $perms, $from_id);
    
    if ($success) {
        tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تمت إضافة مدير عام</b>\n\n👤 ID: <code>{$manager_id}</code>\n🌐 النطاق: جميع البوتات\n🔐 الصلاحيات: {$perms}");
    } else {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل في إضافة المدير");
    }
    break;
}

case '/removemanager': {
    if (count($args) < 2) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /removemanager <label> <user_id>");
        break;
    }
    
    $label = $args[0];
    $manager_id = (int)$args[1];
    
    // الحصول على توكن البوت
    $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
    $stmt->execute([$label]);
    $bot = $stmt->fetch();
    
    if (!$bot) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
        break;
    }
    
    // إزالة المدير
    $success = remove_manager($manager_id, $bot['bot_token']);
    
    if ($success) {
        tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تمت إزالة المدير</b>\n\n👤 ID: <code>{$manager_id}</code>\n🤖 من البوت: <b>{$label}</b>");
    } else {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل في إزالة المدير");
    }
    break;
}

case '/removemanager_all': {
    if (count($args) < 1) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /removemanager_all <user_id>");
        break;
    }
    
    $manager_id = (int)$args[0];
    
    // إزالة المدير من جميع البوتات
    $success = remove_manager($manager_id, null);
    
    if ($success) {
        tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تمت إزالة المدير من جميع البوتات</b>\n\n👤 ID: <code>{$manager_id}</code>");
    } else {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل في إزالة المدير");
    }
    break;
}

case '/managerinfo': {
    if (count($args) < 1) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: /managerinfo <user_id>");
        break;
    }
    
    $manager_id = (int)$args[0];
    
    $managers = $db->prepare("
        SELECT m.*, b.label 
        FROM bot_managers m 
        LEFT JOIN sub_bots b ON m.bot_token = b.bot_token 
        WHERE m.user_id=? AND m.is_active=1
        ORDER BY m.id DESC
    ");
    $managers->execute([$manager_id]);
    $list = $managers->fetchAll();
    
    if (!$list) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ المستخدم <code>{$manager_id}</code> ليس مديراً في النظام");
        break;
    }
    
    $out = "👤 <b>معلومات المدير</b>\n\n";
    $out .= "🆔 <code>{$manager_id}</code>\n";
    if ($list[0]['username']) {
        $out .= "👤 @{$list[0]['username']}\n";
    }
    $out .= "\n<b>الصلاحيات:</b>\n\n";
    
    foreach ($list as $m) {
        $bot_name = $m['label'] ?: '🌐 جميع البوتات';
        $perms_raw = strtolower(trim($m['permissions']));
        
        if ($perms_raw === 'all') {
            $perms_text = '✅ كل الصلاحيات';
        } else {
            $perm_array = explode(':', $perms_raw);
            $perms_list = [];
            foreach ($perm_array as $p) {
                $perms_list[] = permission_to_text((int)$p);
            }
            $perms_text = implode(', ', $perms_list);
        }
        
        $out .= "🤖 <b>{$bot_name}</b>\n";
        $out .= "🔐 {$perms_text}\n";
        $out .= "📅 {$m['created_at']}\n\n";
    }
    
    tg_send($CONTROL_TOKEN, $chat_id, $out);
    break;
}

        // ============================================
        // أوامر نظام المجموعات الجديد
        // ============================================

        case '/group_add': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_add &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            try {
                $db->prepare("INSERT INTO account_groups(group_name, is_default, active) VALUES(?, 0, 1)")
                   ->execute([$group_name]);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إنشاء المجموعة <b>{$group_name}</b>");
            } catch (PDOException $e) {
                if ($e->getCode() == 23000) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> موجودة مسبقاً");
                } else {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
                }
            }
            break;
        }

        case '/group_bind': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_bind &lt;bot_label&gt; &lt;group_name&gt;</code>");
                break;
            }
            
            $bot_label = $args[0];
            $group_name = $args[1];
            
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            try {
                $db->prepare("INSERT IGNORE INTO bot_groups(bot_id, group_id) VALUES(?, ?)")
                   ->execute([$bot['id'], $group['id']]);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم ربط البوت <b>{$bot_label}</b> بالمجموعة <b>{$group_name}</b>");
            } catch (PDOException $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }

        case '/group_unbind': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_unbind &lt;bot_label&gt; &lt;group_name&gt;</code>");
                break;
            }
            
            $bot_label = $args[0];
            $group_name = $args[1];
            
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $db->prepare("DELETE FROM bot_groups WHERE bot_id=? AND group_id=?")
               ->execute([$bot['id'], $group['id']]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم فصل البوت <b>{$bot_label}</b> عن المجموعة <b>{$group_name}</b>");
            break;
        }
        
        case '/group_bind_multi': {
            if (count($args) < 3) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_bind_multi &lt;group_name&gt; &lt;bot1&gt; &lt;bot2&gt; ...</code>\n\n💡 مثال:\n<code>/group_bind_multi 36k test1 test2 test3</code>");
                break;
            }
            
            $group_name = array_shift($args);
            $bot_labels = $args;
            
            // التحقق من المجموعة
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $success = 0;
            $failed = 0;
            $errors = [];
            
            foreach ($bot_labels as $bot_label) {
                $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
                $stmt->execute([$bot_label]);
                $bot = $stmt->fetch();
                
                if (!$bot) {
                    $failed++;
                    $errors[] = "❌ <code>{$bot_label}</code> - غير موجود";
                    continue;
                }
                
                try {
                    $db->prepare("INSERT IGNORE INTO bot_groups(bot_id, group_id) VALUES(?, ?)")
                       ->execute([$bot['id'], $group['id']]);
                    $success++;
                } catch (PDOException $e) {
                    $failed++;
                    $errors[] = "❌ <code>{$bot_label}</code> - خطأ في الربط";
                }
            }
            
            $out = "📊 <b>نتيجة الربط بالمجموعة {$group_name}:</b>\n\n";
            $out .= "✅ نجح: {$success}\n";
            $out .= "❌ فشل: {$failed}\n";
            
            if (!empty($errors)) {
                $out .= "\n<b>التفاصيل:</b>\n" . implode("\n", $errors);
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/group_unbind_multi': {
            if (count($args) < 3) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_unbind_multi &lt;group_name&gt; &lt;bot1&gt; &lt;bot2&gt; ...</code>\n\n💡 مثال:\n<code>/group_unbind_multi 36k test1 test2 test3</code>");
                break;
            }
            
            $group_name = array_shift($args);
            $bot_labels = $args;
            
            // التحقق من المجموعة
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $success = 0;
            $failed = 0;
            $errors = [];
            
            foreach ($bot_labels as $bot_label) {
                $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
                $stmt->execute([$bot_label]);
                $bot = $stmt->fetch();
                
                if (!$bot) {
                    $failed++;
                    $errors[] = "❌ <code>{$bot_label}</code> - غير موجود";
                    continue;
                }
                
                try {
                    $db->prepare("DELETE FROM bot_groups WHERE bot_id=? AND group_id=?")
                       ->execute([$bot['id'], $group['id']]);
                    $success++;
                } catch (PDOException $e) {
                    $failed++;
                    $errors[] = "❌ <code>{$bot_label}</code> - خطأ في الفصل";
                }
            }
            
            $out = "📊 <b>نتيجة فصل البوتات من {$group_name}:</b>\n\n";
            $out .= "✅ نجح: {$success}\n";
            $out .= "❌ فشل: {$failed}\n";
            
            if (!empty($errors)) {
                $out .= "\n<b>التفاصيل:</b>\n" . implode("\n", $errors);
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/group_info': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_info &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $out = "📂 <b>معلومات المجموعة: {$group_name}</b>\n\n";
            $out .= "🆔 ID: {$group['id']}\n";
            $out .= "⭐ افتراضية: " . ($group['is_default'] ? 'نعم' : 'لا') . "\n";
            $out .= "📊 الحالة: " . ($group['active'] ? '✅ نشطة' : '⛔ معطلة') . "\n\n";
            
            $stmt = $db->prepare("
                SELECT sb.label 
                FROM bot_groups bg 
                JOIN sub_bots sb ON sb.id=bg.bot_id 
                WHERE bg.group_id=?
            ");
            $stmt->execute([$group['id']]);
            $bots = $stmt->fetchAll();
            
            $out .= "🤖 <b>البوتات (" . count($bots) . "):</b>\n";
            if ($bots) {
                foreach ($bots as $bot) {
                    $out .= "• {$bot['label']}\n";
                }
            } else {
                $out .= "لا توجد بوتات مرتبطة\n";
            }
            
            $out .= "\n";
            
            $stmt = $db->prepare("
                SELECT a.account_name 
                FROM group_accounts ga 
                JOIN steam_accounts a ON a.id=ga.account_id 
                WHERE ga.group_id=?
                LIMIT 20
            ");
            $stmt->execute([$group['id']]);
            $accounts = $stmt->fetchAll();
            
            $stmt = $db->prepare("SELECT COUNT(*) FROM group_accounts WHERE group_id=?");
            $stmt->execute([$group['id']]);
            $total = $stmt->fetchColumn();
            
            $out .= "👤 <b>الحسابات ({$total}):</b>\n";
            if ($accounts) {
                foreach ($accounts as $acc) {
                    $out .= "• {$acc['account_name']}\n";
                }
                if ($total > 20) {
                    $out .= "... و " . ($total - 20) . " آخرين";
                }
            } else {
                $out .= "لا توجد حسابات";
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/group_account_add': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_account_add &lt;group_name&gt; &lt;acc1&gt; [acc2] [acc3] ...</code>\n\n💡 يمكنك إضافة حساب واحد أو عدة حسابات دفعة واحدة");
                break;
            }
            
            $group_name = $args[0];
            $account_names = array_slice($args, 1);
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $added = 0;
            $skipped = 0;
            $not_found = [];
            
            foreach ($account_names as $account_name) {
                $stmt = $db->prepare("SELECT id FROM steam_accounts WHERE account_name=?");
                $stmt->execute([$account_name]);
                $account = $stmt->fetch();
                
                if (!$account) {
                    $not_found[] = $account_name;
                    continue;
                }
                
                // Check if already exists
                $stmt = $db->prepare("SELECT COUNT(*) as c FROM group_accounts WHERE group_id=? AND account_id=?");
                $stmt->execute([$group['id'], $account['id']]);
                if ($stmt->fetch()['c'] > 0) {
                    $skipped++;
                    continue;
                }
                
                $db->prepare("INSERT INTO group_accounts(group_id, account_id) VALUES(?, ?)")
                   ->execute([$group['id'], $account['id']]);
                $added++;
            }
            
            $out = "📊 <b>ملخص إضافة الحسابات إلى {$group_name}:</b>\n\n";
            $out .= "✅ <b>تم الإضافة:</b> {$added}\n";
            $out .= "⏭️ <b>تم التجاهل (موجود مسبقاً):</b> {$skipped}\n";
            $out .= "❌ <b>غير موجود:</b> " . count($not_found) . "\n";
            
            if ($not_found) {
                $out .= "\n<b>الحسابات غير الموجودة:</b>\n";
                foreach ($not_found as $acc) {
                    $out .= "• {$acc}\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            
            // Log
            $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('group_account_add_batch', ?, ?)")
               ->execute(["Group: {$group_name}, Added: {$added}, Skipped: {$skipped}", $from_id]);
            break;
        }

        case '/group_account_del': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_account_del &lt;group_name&gt; &lt;acc1&gt; [acc2] [acc3] ...</code>\n\n💡 يمكنك حذف حساب واحد أو عدة حسابات دفعة واحدة");
                break;
            }
            
            $group_name = $args[0];
            $account_names = array_slice($args, 1);
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $removed = 0;
            $not_found = [];
            
            foreach ($account_names as $account_name) {
                $stmt = $db->prepare("SELECT id FROM steam_accounts WHERE account_name=?");
                $stmt->execute([$account_name]);
                $account = $stmt->fetch();
                
                if (!$account) {
                    $not_found[] = $account_name;
                    continue;
                }
                
                $stmt = $db->prepare("DELETE FROM group_accounts WHERE group_id=? AND account_id=?");
                $stmt->execute([$group['id'], $account['id']]);
                
                if ($stmt->rowCount() > 0) {
                    $removed++;
                }
            }
            
            $out = "📊 <b>ملخص حذف الحسابات من {$group_name}:</b>\n\n";
            $out .= "✅ <b>تم الحذف:</b> {$removed}\n";
            $out .= "❌ <b>غير موجود:</b> " . count($not_found) . "\n";
            
            if ($not_found) {
                $out .= "\n<b>الحسابات غير الموجودة:</b>\n";
                foreach ($not_found as $acc) {
                    $out .= "• {$acc}\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            
            // Log
            $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('group_account_del_batch', ?, ?)")
               ->execute(["Group: {$group_name}, Removed: {$removed}", $from_id]);
            break;
        }

        case '/groups_list': {
            $groups = $db->query("
                SELECT g.*, 
                       COUNT(DISTINCT bg.bot_id) as bots_count,
                       COUNT(DISTINCT ga.account_id) as accounts_count
                FROM account_groups g
                LEFT JOIN bot_groups bg ON bg.group_id=g.id
                LEFT JOIN group_accounts ga ON ga.group_id=g.id
                GROUP BY g.id
                ORDER BY g.is_default DESC, g.group_name ASC
            ")->fetchAll();
            
            if (!$groups) {
                tg_send($CONTROL_TOKEN, $chat_id, "لا توجد مجموعات");
                break;
            }
            
            $out = "📂 <b>قائمة المجموعات (" . count($groups) . "):</b>\n\n";
            
            foreach ($groups as $g) {
                $status = $g['active'] ? '✅' : '⛔';
                $default_badge = $g['is_default'] ? ' ⭐' : '';
                $out .= "{$status} <b>{$g['group_name']}</b>{$default_badge}\n";
                $out .= "   🤖 {$g['bots_count']} بوت | 👤 {$g['accounts_count']} حساب\n";
                
                // عرض البوتات المرتبطة (الأول 5 فقط)
                if ($g['bots_count'] > 0) {
                    $bots_stmt = $db->prepare("
                        SELECT sb.label 
                        FROM bot_groups bg 
                        JOIN sub_bots sb ON sb.id=bg.bot_id 
                        WHERE bg.group_id=? 
                        LIMIT 5
                    ");
                    $bots_stmt->execute([$g['id']]);
                    $bots = $bots_stmt->fetchAll(PDO::FETCH_COLUMN);
                    $bots_str = implode(', ', $bots);
                    if ($g['bots_count'] > 5) {
                        $bots_str .= ' ...';
                    }
                    $out .= "   └ بوتات: <code>{$bots_str}</code>\n";
                }
                
                $out .= "\n";
            }
            
            // عرض البوتات المرتبطة بأكثر من مجموعة
            $multi_bots = $db->query("
                SELECT sb.label, COUNT(DISTINCT bg.group_id) as groups_count
                FROM bot_groups bg
                JOIN sub_bots sb ON sb.id=bg.bot_id
                GROUP BY bg.bot_id
                HAVING groups_count > 1
                ORDER BY groups_count DESC
            ")->fetchAll();
            
            if (!empty($multi_bots)) {
                $out .= "\n🔗 <b>بوتات في أكثر من مجموعة:</b>\n";
                foreach ($multi_bots as $mb) {
                    $out .= "• <code>{$mb['label']}</code> - {$mb['groups_count']} مجموعات\n";
                }
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/group_toggle': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_toggle &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $new_status = $group['active'] ? 0 : 1;
            $db->prepare("UPDATE account_groups SET active=? WHERE id=?")
               ->execute([$new_status, $group['id']]);
            
            $status_text = $new_status ? 'تفعيل' : 'تعطيل';
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم {$status_text} المجموعة <b>{$group_name}</b>");
            break;
        }

        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        // أوامر المجموعات المتقدمة - NEW
        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

        case '/group_rename': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_rename &lt;old_name&gt; &lt;new_name&gt;</code>");
                break;
            }
            
            $old_name = $args[0];
            $new_name = $args[1];
            
            // Validation
            if (!preg_match('/^[a-z0-9_.-]{1,32}$/i', $new_name)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الاسم الجديد غير صالح. استخدم [a-z0-9_.-] بطول ≤32");
                break;
            }
            
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
            $stmt->execute([$old_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$old_name}</b> غير موجودة");
                break;
            }
            
            // Check if locked
            if (!empty($group['locked'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "🔒 المجموعة <b>{$old_name}</b> مقفلة ولا يمكن إعادة تسميتها\n\nاستخدم <code>/group_unlock {$old_name}</code> أولاً");
                break;
            }
            
            try {
                $db->prepare("UPDATE account_groups SET group_name=? WHERE id=?")
                   ->execute([$new_name, $group['id']]);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إعادة تسمية المجموعة من <b>{$old_name}</b> إلى <b>{$new_name}</b>");
                
                // Log
                $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('group_rename', ?, ?)")
                   ->execute(["$old_name → $new_name", $from_id]);
            } catch (PDOException $e) {
                if ($e->getCode() == 23000) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$new_name}</b> موجودة مسبقاً");
                } else {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
                }
            }
            break;
        }

        case '/group_delete': {
            if (count($args) < 2 || $args[1] !== 'confirm') {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_delete &lt;group_name&gt; confirm</code>\n\n⚠️ هذا الإجراء نهائي ولا يمكن التراجع عنه!");
                break;
            }
            
            $group_name = $args[0];
            
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            // Check if locked
            if (!empty($group['locked'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "🔒 المجموعة <b>{$group_name}</b> مقفلة ولا يمكن حذفها\n\nاستخدم <code>/group_unlock {$group_name}</code> أولاً");
                break;
            }
            
            // Check if default
            if (!empty($group['is_default'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لا يمكن حذف المجموعة الافتراضية");
                break;
            }
            
            // Check bot links
            $stmt = $db->prepare("SELECT COUNT(*) as c FROM bot_groups WHERE group_id=?");
            $stmt->execute([$group['id']]);
            $bot_count = $stmt->fetch()['c'];
            
            if ($bot_count > 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة مرتبطة بـ <b>{$bot_count}</b> بوت\n\n" .
                    "يجب فك الربط أولاً باستخدام:\n<code>/group_unbind &lt;bot&gt; {$group_name}</code>");
                break;
            }
            
            // Delete
            $db->prepare("DELETE FROM account_groups WHERE id=?")
               ->execute([$group['id']]);
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف المجموعة <b>{$group_name}</b> نهائياً");
            
            // Log
            $db->prepare("INSERT INTO system_logs(action, details, user_id) VALUES('group_delete', ?, ?)")
               ->execute([$group_name, $from_id]);
            break;
        }

        case '/group_lock': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_lock &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $db->prepare("UPDATE account_groups SET locked=1 WHERE id=?")
               ->execute([$group['id']]);
            tg_send($CONTROL_TOKEN, $chat_id, "🔒 تم قفل المجموعة <b>{$group_name}</b>\n\nلا يمكن الآن: rename، delete");
            break;
        }

        case '/group_unlock': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_unlock &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            $stmt = $db->prepare("SELECT id FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            $db->prepare("UPDATE account_groups SET locked=0 WHERE id=?")
               ->execute([$group['id']]);
            tg_send($CONTROL_TOKEN, $chat_id, "🔓 تم فتح قفل المجموعة <b>{$group_name}</b>");
            break;
        }

        case '/group_usage': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/group_usage &lt;group_name&gt;</code>");
                break;
            }
            
            $group_name = $args[0];
            
            // Get group (using generated column for faster search)
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name_lower=LOWER(?)");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            // Count bots
            $stmt = $db->prepare("SELECT COUNT(*) AS bots_count FROM bot_groups WHERE group_id=?");
            $stmt->execute([$group['id']]);
            $bots_count = $stmt->fetch()['bots_count'];
            
            // Get accounts
            $stmt = $db->prepare("
                SELECT a.account_name 
                FROM group_accounts ga 
                JOIN steam_accounts a ON a.id=ga.account_id 
                WHERE ga.group_id=?
                ORDER BY a.account_name
            ");
            $stmt->execute([$group['id']]);
            $accounts = $stmt->fetchAll();
            
            $out = "📊 <b>استخدام المجموعة: {$group['group_name']}</b>\n\n";
            $out .= "🤖 <b>عدد البوتات المرتبطة:</b> {$bots_count}\n";
            $out .= "👤 <b>عدد الحسابات:</b> " . count($accounts) . "\n\n";
            
            if ($accounts) {
                $out .= "📋 <b>قائمة الحسابات:</b>\n";
                foreach ($accounts as $acc) {
                    $out .= "• {$acc['account_name']}\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/bot_groups': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/bot_groups &lt;bot_label&gt;</code>");
                break;
            }
            
            $bot_label = $args[0];
            
            // Get bot
            $stmt = $db->prepare("SELECT * FROM sub_bots WHERE LOWER(label)=LOWER(?)");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // Get groups
            $stmt = $db->prepare("
                SELECT g.group_name, g.locked, g.active
                FROM bot_groups bg
                JOIN account_groups g ON g.id=bg.group_id
                WHERE bg.bot_id=?
                ORDER BY g.group_name
            ");
            $stmt->execute([$bot['id']]);
            $groups = $stmt->fetchAll();
            
            // Get visible accounts (UNION)
            $stmt = $db->prepare("
                SELECT DISTINCT a.account_name
                FROM bot_groups bg
                JOIN group_accounts ga ON ga.group_id=bg.group_id
                JOIN steam_accounts a ON a.id=ga.account_id
                WHERE bg.bot_id=?
                ORDER BY a.account_name
            ");
            $stmt->execute([$bot['id']]);
            $accounts = $stmt->fetchAll();
            
            $out = "🤖 <b>مجموعات البوت: {$bot['label']}</b>\n\n";
            $out .= "📂 <b>المجموعات المرتبطة (" . count($groups) . "):</b>\n";
            
            if ($groups) {
                foreach ($groups as $g) {
                    $lock_icon = $g['locked'] ? '🔒' : '';
                    $status_icon = $g['active'] ? '✅' : '⛔';
                    $out .= "• {$status_icon} {$g['group_name']} {$lock_icon}\n";
                }
            } else {
                $out .= "لا توجد مجموعات\n";
            }
            
            $out .= "\n👤 <b>الحسابات المرئية (" . count($accounts) . "):</b>\n";
            if ($accounts) {
                foreach ($accounts as $acc) {
                    $out .= "• {$acc['account_name']}\n";
                }
            } else {
                $out .= "لا توجد حسابات";
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        case '/voucher_who_used': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/voucher_who_used &lt;code&gt;</code>");
                break;
            }
            
            $code = $args[0];
            
            // Get voucher info
            $stmt = $db->prepare("SELECT * FROM vouchers WHERE code=?");
            $stmt->execute([$code]);
            $voucher = $stmt->fetch();
            
            if (!$voucher) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ القسيمة <code>{$code}</code> غير موجودة");
                break;
            }
            
            // Get usage history
            $stmt = $db->prepare("
                SELECT vu.used_at, COALESCE(sb.label, '-') AS bot_label
                FROM voucher_usage vu
                LEFT JOIN sub_bots sb ON sb.id=vu.bot_id
                WHERE vu.voucher_code=?
                ORDER BY vu.used_at DESC
            ");
            $stmt->execute([$code]);
            $usage = $stmt->fetchAll();
            
            $out = "🎫 <b>تقرير القسيمة:</b> <code>{$code}</code>\n\n";
            $out .= "📋 <b>النوع:</b> {$voucher['voucher_type']}\n";
            $out .= "💰 <b>القيمة:</b> {$voucher['amount']}\n";
            $out .= "🎯 <b>الهدف:</b> " . ($voucher['target_label'] ?: 'الكل') . "\n";
            $out .= "🔢 <b>الاستخدامات:</b> {$voucher['current_uses']} / " . ($voucher['max_uses'] ?: '∞') . "\n\n";
            
            if ($usage) {
                $out .= "📊 <b>سجل الاستخدام (" . count($usage) . "):</b>\n\n";
                foreach ($usage as $u) {
                    $out .= "🤖 <b>{$u['bot_label']}</b>\n";
                    $out .= "📅 {$u['used_at']}\n\n";
                }
            } else {
                $out .= "ℹ️ لم تُستخدم بعد";
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        // نظام الحدود الذكي - 13 أمر
        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

        case '/limit_mode_all': {
            if (count($args) < 1 || !in_array($args[0], ['daily', 'weekly', 'monthly'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/limit_mode_all &lt;daily|weekly|monthly&gt;</code>");
                break;
            }
            
            $mode = $args[0];
            
            $stmt = $db->query("SELECT COUNT(*) as c FROM limits_global");
            $exists = $stmt->fetch()['c'] > 0;
            
            if ($exists) {
                $db->prepare("UPDATE limits_global SET mode = ?")->execute([$mode]);
            } else {
                $defaults = [
                    'daily' => ['per_day' => 2, 'weekly_cap' => null, 'monthly_cap' => null, 'ban_days' => null],
                    'weekly' => ['per_day' => 2, 'weekly_cap' => 6, 'monthly_cap' => null, 'ban_days' => 7],
                    'monthly' => ['per_day' => 2, 'weekly_cap' => null, 'monthly_cap' => 20, 'ban_days' => 30]
                ];
                
                $d = $defaults[$mode];
                $db->prepare("INSERT INTO limits_global (mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES (?, ?, ?, ?, ?)")
                   ->execute([$mode, $d['per_day'], $d['weekly_cap'], $d['monthly_cap'], $d['ban_days']]);
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تغيير الوضع العام إلى: <b>{$mode}</b>");
            break;
        }

        case '/limit_mode': {
            if (count($args) < 2 || !in_array($args[1], ['daily', 'weekly', 'monthly'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/limit_mode &lt;bot&gt; &lt;daily|weekly|monthly&gt;</code>");
                break;
            }
            
            $bot_label = $args[0];
            $mode = $args[1];
            
            $stmt = $db->prepare("SELECT label FROM sub_bots WHERE label = ?");
            $stmt->execute([$bot_label]);
            if (!$stmt->fetch()) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            $defaults = [
                'daily' => ['per_day' => 2, 'weekly_cap' => null, 'monthly_cap' => null, 'ban_days' => null],
                'weekly' => ['per_day' => 2, 'weekly_cap' => 6, 'monthly_cap' => null, 'ban_days' => 7],
                'monthly' => ['per_day' => 2, 'weekly_cap' => null, 'monthly_cap' => 20, 'ban_days' => 30]
            ];
            
            $d = $defaults[$mode];
            
            $db->prepare("INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE mode = VALUES(mode), per_day = VALUES(per_day), weekly_cap = VALUES(weekly_cap), monthly_cap = VALUES(monthly_cap), ban_days = VALUES(ban_days)")
               ->execute([$bot_label, $mode, $d['per_day'], $d['weekly_cap'], $d['monthly_cap'], $d['ban_days']]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعيين وضع <b>{$mode}</b> للبوت <b>{$bot_label}</b>");
            break;
        }

        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        // حدود البوتات المتقدمة (Bot-Specific Limits) - NEW
        // ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        
        case '/set_daily': {
            if (count($args) < 2 || !is_numeric($args[1])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ <b>الاستخدام:</b>\n<code>/set_daily &lt;bot&gt; &lt;per_day&gt;</code>\n\nمثال: <code>/set_daily testbot 3</code>");
                break;
            }
            
            $bot_label = $args[0];
            $per_day = (int)$args[1];
            
            if ($per_day < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد اليومي يجب أن يكون 1 على الأقل");
                break;
            }
            
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label = ?");
            $stmt->execute([$bot_label]);
            if (!$stmt->fetch()) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت '<b>$bot_label</b>' غير موجود!");
                break;
            }
            
            $db->prepare("INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES (?, 'daily', ?, NULL, NULL, NULL) ON DUPLICATE KEY UPDATE mode = 'daily', per_day = VALUES(per_day), weekly_cap = NULL, monthly_cap = NULL, ban_days = NULL")->execute([$bot_label, $per_day]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تم تعيين حد يومي للبوت '$bot_label'</b>\n\n📊 الحدود:\n• 📅 <b>يومي:</b> $per_day كود\n• 🚫 <b>الحظر:</b> بدون حظر (رفض فوري)\n\n💡 للتحقق: <code>/show_limit $bot_label</code>");
            break;
        }
        
        case '/set_weekly': {
            if (count($args) < 4 || !is_numeric($args[1]) || !is_numeric($args[2]) || !is_numeric($args[3])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ <b>الاستخدام:</b>\n<code>/set_weekly &lt;bot&gt; &lt;per_day&gt; &lt;weekly_cap&gt; &lt;ban_days&gt;</code>\n\nمثال: <code>/set_weekly vipbot 5 20 3</code>");
                break;
            }
            
            $bot_label = $args[0];
            $per_day = (int)$args[1];
            $weekly_cap = (int)$args[2];
            $ban_days = (int)$args[3];
            
            if ($per_day < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد اليومي يجب أن يكون 1 على الأقل");
                break;
            }
            
            if ($weekly_cap < $per_day) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد الأسبوعي ($weekly_cap) يجب أن يكون أكبر من أو يساوي الحد اليومي ($per_day)");
                break;
            }
            
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label = ?");
            $stmt->execute([$bot_label]);
            if (!$stmt->fetch()) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت '<b>$bot_label</b>' غير موجود!");
                break;
            }
            
            $db->prepare("INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES (?, 'weekly', ?, ?, NULL, ?) ON DUPLICATE KEY UPDATE mode = 'weekly', per_day = VALUES(per_day), weekly_cap = VALUES(weekly_cap), monthly_cap = NULL, ban_days = VALUES(ban_days)")->execute([$bot_label, $per_day, $weekly_cap, $ban_days]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تم تعيين حد أسبوعي للبوت '$bot_label'</b>\n\n📊 الحدود:\n• 📅 <b>يومي:</b> $per_day كود\n• 📆 <b>أسبوعي:</b> $weekly_cap كود\n• 🚫 <b>حظر:</b> $ban_days أيام\n\n💡 للتحقق: <code>/show_limit $bot_label</code>");
            break;
        }
        
        case '/set_monthly': {
            if (count($args) < 4 || !is_numeric($args[1]) || !is_numeric($args[2]) || !is_numeric($args[3])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ <b>الاستخدام:</b>\n<code>/set_monthly &lt;bot&gt; &lt;per_day&gt; &lt;monthly_cap&gt; &lt;ban_days&gt;</code>\n\nمثال: <code>/set_monthly premiumbot 10 100 7</code>");
                break;
            }
            
            $bot_label = $args[0];
            $per_day = (int)$args[1];
            $monthly_cap = (int)$args[2];
            $ban_days = (int)$args[3];
            
            if ($per_day < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد اليومي يجب أن يكون 1 على الأقل");
                break;
            }
            
            if ($monthly_cap < $per_day) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد الشهري ($monthly_cap) يجب أن يكون أكبر من أو يساوي الحد اليومي ($per_day)");
                break;
            }
            
            $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label = ?");
            $stmt->execute([$bot_label]);
            if (!$stmt->fetch()) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت '<b>$bot_label</b>' غير موجود!");
                break;
            }
            
            $db->prepare("INSERT INTO limits_bot (bot_label, mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES (?, 'monthly', ?, NULL, ?, ?) ON DUPLICATE KEY UPDATE mode = 'monthly', per_day = VALUES(per_day), weekly_cap = NULL, monthly_cap = VALUES(monthly_cap), ban_days = VALUES(ban_days)")->execute([$bot_label, $per_day, $monthly_cap, $ban_days]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تم تعيين حد شهري للبوت '$bot_label'</b>\n\n📊 الحدود:\n• 📅 <b>يومي:</b> $per_day كود\n• 📆 <b>شهري:</b> $monthly_cap كود\n• 🚫 <b>حظر:</b> $ban_days أيام\n\n💡 للتحقق: <code>/show_limit $bot_label</code>");
            break;
        }
        
        case '/bot_limit_clear': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ <b>الاستخدام:</b>\n<code>/bot_limit_clear &lt;bot_label&gt;</code>\n\nمثال: <code>/bot_limit_clear testbot</code>");
                break;
            }
            
            $bot_label = $args[0];
            
            $stmt = $db->prepare("DELETE FROM limits_bot WHERE bot_label = ?");
            $stmt->execute([$bot_label]);
            
            if ($stmt->rowCount() > 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف الحدود الخاصة للبوت '<b>$bot_label</b>'\n\n🌍 سيستخدم البوت الآن الحدود العامة\n\n💡 للتحقق: <code>/show_limit $bot_label</code>");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "ℹ️ البوت '<b>$bot_label</b>' لم يكن لديه حدود خاصة\n\n🌍 يستخدم الحدود العامة حالياً");
            }
            break;
        }

        case '/set_daily_all': {
            if (count($args) < 1 || !is_numeric($args[0])) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/set_daily_all &lt;per_day&gt;</code>\nمثال: <code>/set_daily_all 3</code>");
                break;
            }
            
            $per_day = (int)$args[0];
            
            // تحديث أو إدخال
            $stmt = $db->query("SELECT COUNT(*) as c FROM limits_global");
            $exists = $stmt->fetch()['c'] > 0;
            
            if ($exists) {
                $db->prepare("UPDATE limits_global SET mode = 'daily', per_day = ?, weekly_cap = NULL, monthly_cap = NULL, ban_days = NULL")
                   ->execute([$per_day]);
            } else {
                $db->prepare("INSERT INTO limits_global (mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES ('daily', ?, NULL, NULL, NULL)")
                   ->execute([$per_day]);
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعيين الحد اليومي العام: <b>{$per_day}</b> كود/يوم");
            break;
        }

        case '/set_weekly_all': {
            if (count($args) < 3 || !is_numeric($args[0]) || !is_numeric($args[1]) || !is_numeric($args[2])) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/set_weekly_all &lt;per_day&gt; &lt;weekly_cap&gt; &lt;ban_days&gt;</code>\nمثال: <code>/set_weekly_all 2 6 7</code>");
                break;
            }
            
            $per_day = (int)$args[0];
            $weekly_cap = (int)$args[1];
            $ban_days = (int)$args[2];
            
            // تحديث أو إدخال
            $stmt = $db->query("SELECT COUNT(*) as c FROM limits_global");
            $exists = $stmt->fetch()['c'] > 0;
            
            if ($exists) {
                $db->prepare("UPDATE limits_global SET mode = 'weekly', per_day = ?, weekly_cap = ?, monthly_cap = NULL, ban_days = ?")
                   ->execute([$per_day, $weekly_cap, $ban_days]);
            } else {
                $db->prepare("INSERT INTO limits_global (mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES ('weekly', ?, ?, NULL, ?)")
                   ->execute([$per_day, $weekly_cap, $ban_days]);
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعيين الحد الأسبوعي العام:\n• <b>{$per_day}</b> كود/يوم\n• <b>{$weekly_cap}</b> كود/أسبوع\n• حظر <b>{$ban_days}</b> أيام عند التجاوز");
            break;
        }

        case '/set_monthly_all': {
            if (count($args) < 3 || !is_numeric($args[0]) || !is_numeric($args[1]) || !is_numeric($args[2])) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/set_monthly_all &lt;per_day&gt; &lt;monthly_cap&gt; &lt;ban_days&gt;</code>\nمثال: <code>/set_monthly_all 2 20 30</code>");
                break;
            }
            
            $per_day = (int)$args[0];
            $monthly_cap = (int)$args[1];
            $ban_days = (int)$args[2];
            
            // تحديث أو إدخال
            $stmt = $db->query("SELECT COUNT(*) as c FROM limits_global");
            $exists = $stmt->fetch()['c'] > 0;
            
            if ($exists) {
                $db->prepare("UPDATE limits_global SET mode = 'monthly', per_day = ?, weekly_cap = NULL, monthly_cap = ?, ban_days = ?")
                   ->execute([$per_day, $monthly_cap, $ban_days]);
            } else {
                $db->prepare("INSERT INTO limits_global (mode, per_day, weekly_cap, monthly_cap, ban_days) VALUES ('monthly', ?, NULL, ?, ?)")
                   ->execute([$per_day, $monthly_cap, $ban_days]);
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تعيين الحد الشهري العام:\n• <b>{$per_day}</b> كود/يوم\n• <b>{$monthly_cap}</b> كود/شهر\n• حظر <b>{$ban_days}</b> يوم عند التجاوز");
            break;
        }

        case '/override_account': {
            if (count($args) < 3) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/override_account &lt;account_id&gt; &lt;mode&gt; &lt;per_day&gt; [cap] [ban_days]</code>\n\nمثال يومي: <code>/override_account 5 daily 5</code>\nمثال أسبوعي: <code>/override_account 5 weekly 3 10 7</code>");
                break;
            }
            
            $account_id = (int)$args[0];
            $mode = $args[1];
            $per_day = (int)$args[2];
            $cap = isset($args[3]) ? (int)$args[3] : null;
            $ban_days = isset($args[4]) ? (int)$args[4] : null;
            
            if (!in_array($mode, ['daily', 'weekly', 'monthly'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الوضع يجب أن يكون: daily أو weekly أو monthly");
                break;
            }
            
            $stmt = $db->prepare("SELECT account_name FROM steam_accounts WHERE id = ?");
            $stmt->execute([$account_id]);
            $acc = $stmt->fetch();
            
            if (!$acc) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحساب ID={$account_id} غير موجود");
                break;
            }
            
            $db->prepare("INSERT INTO limits_override (account_id, mode, per_day, cap, ban_days) VALUES (?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE mode = VALUES(mode), per_day = VALUES(per_day), cap = VALUES(cap), ban_days = VALUES(ban_days)")
               ->execute([$account_id, $mode, $per_day, $cap, $ban_days]);
            
            $msg = "✅ تم تعيين حدود خاصة للحساب <code>{$acc['account_name']}</code>:\n• الوضع: <b>{$mode}</b>\n• <b>{$per_day}</b> كود/يوم";
            
            if ($cap) {
                $period = $mode === 'weekly' ? 'أسبوع' : 'شهر';
                $msg .= "\n• <b>{$cap}</b> كود/{$period}";
            }
            
            if ($ban_days) {
                $msg .= "\n• حظر <b>{$ban_days}</b> يوم عند التجاوز";
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $msg);
            break;
        }

        case '/override_clear': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/override_clear &lt;account_id&gt;</code>");
                break;
            }
            
            $account_id = (int)$args[0];
            
            $stmt = $db->prepare("DELETE FROM limits_override WHERE account_id = ?");
            $stmt->execute([$account_id]);
            
            if ($stmt->rowCount() > 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف الحدود الخاصة للحساب ID={$account_id}");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد حدود خاصة لهذا الحساب");
            }
            break;
        }

        case '/show_limit': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ <b>الاستخدام:</b>\n\n" .
                       "<code>/show_limit global</code> - الحدود العامة\n" .
                       "<code>/show_limit bot</code> - جميع البوتات\n" .
                       "<code>/show_limit bot/&lt;label&gt;</code> - بوت محدد\n" .
                       "<code>/show_limit &lt;label&gt;</code> - بوت محدد\n" .
                       "<code>/show_limit &lt;account_id&gt;</code> - حساب محدد\n\n" .
                       "مثال: <code>/show_limit bot/vipbot</code>");
                break;
            }
            
            $target = $args[0];
            
            // --- عرض الحدود العامة ---
            if ($target === 'global') {
                $stmt = $db->query("SELECT * FROM limits_global LIMIT 1");
                $limit = $stmt->fetch();
                
                if (!$limit) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد حدود عامة مُعرّفة");
                    break;
                }
                
                $mode_ar = ['daily' => '📅 يومي', 'weekly' => '📆 أسبوعي', 'monthly' => '📆 شهري'][$limit['mode']];
                
                $msg = "🌍 <b>الحدود العامة (Global Limits)</b>\n\n";
                $msg .= "📊 الوضع: $mode_ar\n";
                $msg .= "📅 الحد اليومي: <b>{$limit['per_day']}</b> كود\n";
                
                if ($limit['weekly_cap']) {
                    $msg .= "📆 السقف الأسبوعي: <b>{$limit['weekly_cap']}</b> كود\n";
                }
                
                if ($limit['monthly_cap']) {
                    $msg .= "📆 السقف الشهري: <b>{$limit['monthly_cap']}</b> كود\n";
                }
                
                if ($limit['ban_days']) {
                    $msg .= "🚫 مدة الحظر: <b>{$limit['ban_days']}</b> يوم\n";
                } else {
                    $msg .= "🚫 مدة الحظر: <b>بدون حظر</b>\n";
                }
                
                $msg .= "\n💡 <i>هذه الحدود تُطبق على جميع البوتات ما لم يكن لديها حدود خاصة</i>";
                
                tg_send($CONTROL_TOKEN, $chat_id, $msg);
                break;
            }
            
            // --- عرض حدود جميع البوتات ---
            if ($target === 'bot') {
                $stmt = $db->query("
                    SELECT 
                        sb.label,
                        lb.mode,
                        lb.per_day,
                        lb.weekly_cap,
                        lb.monthly_cap,
                        lb.ban_days,
                        CASE WHEN lb.bot_label IS NOT NULL THEN 1 ELSE 0 END as has_custom
                    FROM sub_bots sb
                    LEFT JOIN limits_bot lb ON lb.bot_label = sb.label
                    ORDER BY has_custom DESC, sb.label
                ");
                
                $bots = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (empty($bots)) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد بوتات مسجلة!");
                    break;
                }
                
                $msg = "🤖 <b>حدود البوتات (Bot Limits)</b>\n\n";
                
                $with_custom = array_filter($bots, fn($b) => $b['has_custom']);
                $without_custom = array_filter($bots, fn($b) => !$b['has_custom']);
                
                if (!empty($with_custom)) {
                    $msg .= "✅ <b>بوتات لديها حدود خاصة:</b>\n\n";
                    
                    foreach ($with_custom as $bot) {
                        $mode_ar = ['daily' => '📅 يومي', 'weekly' => '📆 أسبوعي', 'monthly' => '📆 شهري'][$bot['mode']];
                        
                        $msg .= "• <b>{$bot['label']}</b>\n";
                        $msg .= "  $mode_ar: {$bot['per_day']} كود/يوم";
                        
                        if ($bot['mode'] === 'weekly' && $bot['weekly_cap']) {
                            $msg .= ", {$bot['weekly_cap']}/أسبوع";
                        }
                        if ($bot['mode'] === 'monthly' && $bot['monthly_cap']) {
                            $msg .= ", {$bot['monthly_cap']}/شهر";
                        }
                        if ($bot['ban_days']) {
                            $msg .= ", 🚫 {$bot['ban_days']} أيام";
                        }
                        
                        $msg .= "\n";
                    }
                }
                
                if (!empty($without_custom)) {
                    $msg .= "\n🌍 <b>بوتات تستخدم الحدود العامة:</b>\n";
                    foreach ($without_custom as $bot) {
                        $msg .= "• {$bot['label']}\n";
                    }
                    $msg .= "\n💡 <i>لتخصيص حدود: /set_weekly bot per_day cap ban</i>";
                }
                
                tg_send($CONTROL_TOKEN, $chat_id, $msg);
                break;
            }
            
            // --- التحقق من صيغة bot/label ---
            $bot_label = null;
            if (strpos($target, 'bot/') === 0) {
                $bot_label = substr($target, 4);
            } elseif (!is_numeric($target)) {
                $bot_label = $target;
            }
            
            // --- عرض حدود بوت محدد ---
            if ($bot_label) {
                // التحقق من وجود البوت
                $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label = ?");
                $stmt->execute([$bot_label]);
                if (!$stmt->fetch()) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت '<b>$bot_label</b>' غير موجود!");
                    break;
                }
                
                // جلب الحدود الخاصة
                $stmt = $db->prepare("SELECT * FROM limits_bot WHERE bot_label = ?");
                $stmt->execute([$bot_label]);
                $bot_limit = $stmt->fetch(PDO::FETCH_ASSOC);
                
                $msg = "🤖 <b>حدود البوت: $bot_label</b>\n\n";
                
                if ($bot_limit) {
                    $mode_ar = ['daily' => 'يومي', 'weekly' => 'أسبوعي', 'monthly' => 'شهري'][$bot_limit['mode']];
                    
                    $msg .= "✅ <b>حدود خاصة:</b>\n\n";
                    $msg .= "📊 الوضع: <code>$mode_ar</code>\n";
                    $msg .= "📅 الحد اليومي: <code>{$bot_limit['per_day']}</code> كود\n";
                    
                    if ($bot_limit['mode'] === 'weekly' && $bot_limit['weekly_cap']) {
                        $msg .= "📆 السقف الأسبوعي: <code>{$bot_limit['weekly_cap']}</code> كود\n";
                    }
                    
                    if ($bot_limit['mode'] === 'monthly' && $bot_limit['monthly_cap']) {
                        $msg .= "📆 السقف الشهري: <code>{$bot_limit['monthly_cap']}</code> كود\n";
                    }
                    
                    if ($bot_limit['ban_days']) {
                        $msg .= "🚫 مدة الحظر: <code>{$bot_limit['ban_days']}</code> أيام\n";
                    } else {
                        $msg .= "🚫 مدة الحظر: <code>بدون حظر</code>\n";
                    }
                    
                    $msg .= "\n💡 <i>لحذف الحدود الخاصة: </i><code>/bot_limit_clear $bot_label</code>";
                    
                } else {
                    // جلب الحدود العامة
                    $stmt = $db->query("SELECT * FROM limits_global LIMIT 1");
                    $global = $stmt->fetch();
                    
                    $mode_ar = ['daily' => 'يومي', 'weekly' => 'أسبوعي', 'monthly' => 'شهري'][$global['mode']];
                    
                    $msg .= "🌍 <b>يستخدم الحدود العامة:</b>\n\n";
                    $msg .= "📊 الوضع: <code>$mode_ar</code>\n";
                    $msg .= "📅 الحد اليومي: <code>{$global['per_day']}</code> كود\n";
                    
                    if ($global['mode'] === 'weekly' && $global['weekly_cap']) {
                        $msg .= "📆 السقف الأسبوعي: <code>{$global['weekly_cap']}</code> كود\n";
                    }
                    
                    if ($global['mode'] === 'monthly' && $global['monthly_cap']) {
                        $msg .= "📆 السقف الشهري: <code>{$global['monthly_cap']}</code> كود\n";
                    }
                    
                    if ($global['ban_days']) {
                        $msg .= "🚫 مدة الحظر: <code>{$global['ban_days']}</code> أيام\n";
                    }
                    
                    $msg .= "\n💡 <i>لتخصيص حدود لهذا البوت: </i><code>/set_weekly $bot_label per_day cap ban</code>";
                }
                
                tg_send($CONTROL_TOKEN, $chat_id, $msg);
                break;
            }
            
            // --- عرض حدود حساب محدد ---
            if (is_numeric($target)) {
                $account_id = (int)$target;
                
                $stmt = $db->prepare("SELECT lo.*, sa.account_name FROM limits_override lo JOIN steam_accounts sa ON sa.id = lo.account_id WHERE lo.account_id = ?");
                $stmt->execute([$account_id]);
                $limit = $stmt->fetch();
                
                if (!$limit) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد حدود خاصة للحساب ID={$account_id}");
                    break;
                }
                
                $mode_ar = ['daily' => 'يومي', 'weekly' => 'أسبوعي', 'monthly' => 'شهري'][$limit['mode']];
                
                $msg = "🎮 <b>حدود الحساب: {$limit['account_name']}</b>\n";
                $msg .= "🆔 ID: <code>$account_id</code>\n\n";
                $msg .= "⭐ <b>حدود مخصصة (Override):</b>\n\n";
                $msg .= "📊 الوضع: <code>$mode_ar</code>\n";
                $msg .= "📅 الحد اليومي: <code>{$limit['per_day']}</code> كود\n";
                
                if ($limit['cap']) {
                    $period = $limit['mode'] === 'weekly' ? 'أسبوع' : 'شهر';
                    $msg .= "📆 السقف: <code>{$limit['cap']}</code> كود/{$period}\n";
                }
                
                if ($limit['ban_days']) {
                    $msg .= "🚫 مدة الحظر: <code>{$limit['ban_days']}</code> يوم\n";
                }
                
                tg_send($CONTROL_TOKEN, $chat_id, $msg);
            }
            
            break;
        }

        case '/reset_counters': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/reset_counters &lt;user_id|all&gt;</code>\n\nأمثلة:\n• <code>/reset_counters 123456789</code>\n• <code>/reset_counters all</code>");
                break;
            }
            
            $target = $args[0];
            
            if ($target === 'all') {
                $stmt = $db->query("UPDATE limit_counters SET day_used = 0, week_used = 0, month_used = 0");
                $count = $stmt->rowCount();
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إعادة تعيين <b>{$count}</b> عداد");
            }
            elseif (is_numeric($target)) {
                $user_id = (int)$target;
                $stmt = $db->prepare("UPDATE limit_counters SET day_used = 0, week_used = 0, month_used = 0 WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $count = $stmt->rowCount();
                
                if ($count > 0) {
                    tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إعادة تعيين <b>{$count}</b> عداد للمستخدم {$user_id}");
                } else {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد عدادات لهذا المستخدم");
                }
            }
            else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ يجب تحديد user_id أو 'all'");
            }
            break;
        }

        case '/ban_from_account': {
            if (count($args) < 3) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/ban_from_account &lt;user_id&gt; &lt;account_id&gt; &lt;days&gt;</code>\nمثال: <code>/ban_from_account 123456789 5 7</code>");
                break;
            }
            
            $user_id = (int)$args[0];
            $account_id = (int)$args[1];
            $days = (int)$args[2];
            
            require_once __DIR__ . '/lib/limits.php';
            apply_ban($db, $user_id, $account_id, $days, 'manual');
            
            $banned_until = date('d/m/Y H:i', strtotime("+{$days} days"));
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حظر المستخدم <code>{$user_id}</code> من الحساب ID={$account_id}\nالمدة: <b>{$days}</b> يوم\nحتى: {$banned_until}");
            break;
        }

        case '/unban_from_account': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/unban_from_account &lt;user_id&gt; &lt;account_id&gt;</code>\nمثال: <code>/unban_from_account 123456789 5</code>");
                break;
            }
            
            $user_id = (int)$args[0];
            $account_id = (int)$args[1];
            
            require_once __DIR__ . '/lib/limits.php';
            $result = unban($db, $user_id, $account_id);
            
            if ($result) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إلغاء حظر المستخدم <code>{$user_id}</code> من الحساب ID={$account_id}");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لا يوجد حظر نشط لهذا المستخدم");
            }
            break;
        }

        case '/msg_set': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/msg_set &lt;event&gt; &lt;text&gt;</code>\n\nالأحداث المتاحة:\n• on_share_warning\n• on_daily_near\n• on_limit_hit\n• on_banned\n• on_unbanned\n\nمثال:\n<code>/msg_set on_limit_hit وصلت للحد الأقصى اليوم!</code>");
                break;
            }
            
            $event = $args[0];
            $text = implode(' ', array_slice($args, 1));
            
            $valid_events = ['on_share_warning', 'on_daily_near', 'on_limit_hit', 'on_banned', 'on_unbanned'];
            
            if (!in_array($event, $valid_events)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ حدث غير صالح. الأحداث المتاحة:\n" . implode("\n", $valid_events));
                break;
            }
            
            $db->prepare("UPDATE limit_messages SET text = ? WHERE event = ?")
               ->execute([$text, $event]);
            
            tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تحديث رسالة الحدث <code>{$event}</code>");
            break;
        }

        case '/msg_show': {
            try {
                $stmt = $db->query("SELECT event, text FROM limit_messages ORDER BY event");
                $messages = $stmt->fetchAll();
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ في قاعدة البيانات:\n" . $e->getMessage());
                break;
            }
            
            if (empty($messages)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ لا توجد رسائل مُعرّفة\n\n💡 تأكد من تطبيق SQL:\n<code>mysql -u user -p db < sql/supehgku_test0.sql</code>");
                break;
            }
            
            $msg = "📝 <b>رسائل الأحداث:</b>\n\n";
            
            foreach ($messages as $m) {
                $preview = mb_substr($m['text'], 0, 50);
                if (mb_strlen($m['text']) > 50) {
                    $preview .= '...';
                }
                // هروب HTML entities لتجنب التضارب
                $preview_escaped = htmlspecialchars($preview, ENT_QUOTES, 'UTF-8');
                
                $msg .= "🔹 <code>{$m['event']}</code>\n{$preview_escaped}\n\n";
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $msg);
            break;
        }

        // ════════════════════════════════════════════════════════════
        // الأوامر الجديدة - إدارة متقدمة
        // ════════════════════════════════════════════════════════════

        case '/bot_rename': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ <b>الاستخدام:</b>\n" .
                    "<code>/bot_rename &lt;label_قديم&gt; &lt;label_جديد&gt;</code>\n\n" .
                    "💡 <i>مثال:</i> <code>/bot_rename test1 test_new</code>");
                break;
            }
            
            $old_label = $args[0];
            $new_label = $args[1];
            
            // التحقق من البوت القديم
            $stmt = $db->prepare("SELECT * FROM sub_bots WHERE label=?");
            $stmt->execute([$old_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$old_label}</b> غير موجود");
                break;
            }
            
            // التحقق من عدم وجود الاسم الجديد
            $stmt = $db->prepare("SELECT * FROM sub_bots WHERE label=?");
            $stmt->execute([$new_label]);
            if ($stmt->fetch()) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الاسم <b>{$new_label}</b> مستخدم بالفعل");
                break;
            }
            
            try {
                // تحديث اسم البوت
                $stmt = $db->prepare("UPDATE sub_bots SET label=? WHERE label=?");
                $stmt->execute([$new_label, $old_label]);
                
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "✅ <b>تم تعديل اسم البوت بنجاح!</b>\n\n" .
                    "الاسم القديم: <code>{$old_label}</code>\n" .
                    "الاسم الجديد: <code>{$new_label}</code>\n\n" .
                    "⚠️ <b>ملاحظة:</b> سيتم تحديث:\n" .
                    "• الحدود الخاصة بالبوت\n" .
                    "• العدادات المرتبطة\n" .
                    "• الحظر المرتبط\n" .
                    "تلقائياً (Foreign Key Cascade)");
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            
            break;
        }

        case '/group_details': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ <b>الاستخدام:</b>\n" .
                    "<code>/group_details &lt;group_name&gt;</code>\n\n" .
                    "💡 <i>مثال:</i> <code>/group_details default</code>");
                break;
            }
            
            $group_name = $args[0];
            
            // الحصول على المجموعة
            $stmt = $db->prepare("SELECT * FROM account_groups WHERE group_name=?");
            $stmt->execute([$group_name]);
            $group = $stmt->fetch();
            
            if (!$group) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المجموعة <b>{$group_name}</b> غير موجودة");
                break;
            }
            
            // الحصول على البوتات المرتبطة
            $stmt = $db->prepare("
                SELECT sb.id, sb.label, sb.is_paused
                FROM bot_groups bg
                JOIN sub_bots sb ON sb.id = bg.bot_id
                WHERE bg.group_id = ?
                ORDER BY sb.label ASC
            ");
            $stmt->execute([$group['id']]);
            $bots = $stmt->fetchAll();
            
            // الحصول على الحسابات المرتبطة
            $stmt = $db->prepare("
                SELECT sa.id, sa.account_name, sa.active
                FROM group_accounts ga
                JOIN steam_accounts sa ON sa.id = ga.account_id
                WHERE ga.group_id = ?
                ORDER BY sa.account_name ASC
            ");
            $stmt->execute([$group['id']]);
            $accounts = $stmt->fetchAll();
            
            // بناء الرد
            $status = $group['active'] ? '✅ نشطة' : '⛔ معطلة';
            $default = $group['is_default'] ? ' ⭐ (افتراضية)' : '';
            $group_name_safe = htmlspecialchars($group_name, ENT_QUOTES, 'UTF-8');
            $description_safe = htmlspecialchars($group['description'] ?: 'لا يوجد', ENT_QUOTES, 'UTF-8');
            
            $out = "📂 <b>معلومات المجموعة: {$group_name_safe}</b>{$default}\n\n";
            $out .= "🔹 <b>الحالة:</b> {$status}\n";
            $out .= "🔹 <b>الوصف:</b> {$description_safe}\n\n";
            
            // البوتات
            $out .= "🤖 <b>البوتات المرتبطة:</b> (" . count($bots) . ")\n";
            if ($bots) {
                foreach ($bots as $i => $bot) {
                    $num = $i + 1;
                    $status_icon = $bot['is_paused'] ? '⏸️' : '▶️';
                    $bot_label_safe = htmlspecialchars($bot['label'], ENT_QUOTES, 'UTF-8');
                    $out .= "{$num}. {$status_icon} <code>{$bot_label_safe}</code> (ID: {$bot['id']})\n";
                }
            } else {
                $out .= "   <i>لا توجد بوتات مرتبطة</i>\n";
            }
            
            $out .= "\n👤 <b>الحسابات المرتبطة:</b> (" . count($accounts) . ")\n";
            if ($accounts) {
                foreach ($accounts as $i => $acc) {
                    $num = $i + 1;
                    $status_icon = $acc['active'] ? '✅' : '❌';
                    $acc_name_safe = htmlspecialchars($acc['account_name'], ENT_QUOTES, 'UTF-8');
                    $out .= "{$num}. {$status_icon} <code>{$acc_name_safe}</code> (ID: {$acc['id']})\n";
                }
            } else {
                $out .= "   <i>لا توجد حسابات مرتبطة</i>\n";
            }
            
            $send_result = tg_send($CONTROL_TOKEN, $chat_id, $out);
            if (!($send_result['ok'] ?? false)) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ فشل إرسال التفاصيل. الخطأ:\n" .
                    "<code>" . htmlspecialchars($send_result['description'] ?? 'unknown', ENT_QUOTES, 'UTF-8') . "</code>\n\n" .
                    "طول الرسالة: " . strlen($out) . " حرف"
                );
            }
            break;
        }

        case '/bot_details': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ <b>الاستخدام:</b>\n" .
                    "<code>/bot_details &lt;bot_label&gt;</code>\n\n" .
                    "💡 <i>مثال:</i> <code>/bot_details test1</code>");
                break;
            }
            
            $bot_label = $args[0];
            
            // الحصول على البوت
            $stmt = $db->prepare("SELECT * FROM sub_bots WHERE label=?");
            $stmt->execute([$bot_label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                break;
            }
            
            // الحصول على المجموعات المرتبطة
            $stmt = $db->prepare("
                SELECT ag.id, ag.group_name, ag.active
                FROM bot_groups bg
                JOIN account_groups ag ON ag.id = bg.group_id
                WHERE bg.bot_id = ?
                ORDER BY ag.group_name ASC
            ");
            $stmt->execute([$bot['id']]);
            $groups = $stmt->fetchAll();
            
            // الحصول على الحسابات المرتبطة (عبر المجموعات)
            $stmt = $db->prepare("
                SELECT DISTINCT sa.id, sa.account_name, sa.active, ag.group_name
                FROM bot_groups bg
                JOIN group_accounts ga ON ga.group_id = bg.group_id
                JOIN steam_accounts sa ON sa.id = ga.account_id
                JOIN account_groups ag ON ag.id = ga.group_id
                WHERE bg.bot_id = ?
                ORDER BY ag.group_name, sa.account_name ASC
            ");
            $stmt->execute([$bot['id']]);
            $accounts = $stmt->fetchAll();
            
            // الحصول على الحدود الخاصة بالبوت
            $stmt = $db->prepare("SELECT * FROM limits_bot WHERE bot_label=?");
            $stmt->execute([$bot_label]);
            $limits = $stmt->fetch();
            
            // بناء الرد
            $status = $bot['is_paused'] ? '⏸️ متوقف' : '▶️ نشط';
            $bot_label_safe = htmlspecialchars($bot_label, ENT_QUOTES, 'UTF-8');
            
            $out = "🤖 <b>معلومات البوت: {$bot_label_safe}</b>\n\n";
            $out .= "🔹 <b>ID:</b> {$bot['id']}\n";
            $out .= "🔹 <b>الحالة:</b> {$status}\n";
            $out .= "🔹 <b>الحد الأقصى للمستخدمين:</b> {$bot['max_users']}\n\n";
            
            // المجموعات
            $out .= "📂 <b>المجموعات المرتبطة:</b> (" . count($groups) . ")\n";
            if ($groups) {
                foreach ($groups as $i => $group) {
                    $num = $i + 1;
                    $status_icon = $group['active'] ? '✅' : '⛔';
                    $group_name_safe = htmlspecialchars($group['group_name'], ENT_QUOTES, 'UTF-8');
                    $out .= "{$num}. {$status_icon} <code>{$group_name_safe}</code> (ID: {$group['id']})\n";
                }
            } else {
                $out .= "   <i>لا توجد مجموعات مرتبطة</i>\n";
            }
            
            // الحسابات
            $out .= "\n👤 <b>الحسابات المرتبطة:</b> (" . count($accounts) . ")\n";
            if ($accounts) {
                $current_group = '';
                $num = 1;
                foreach ($accounts as $acc) {
                    if ($current_group !== $acc['group_name']) {
                        $current_group = $acc['group_name'];
                        $current_group_safe = htmlspecialchars($current_group, ENT_QUOTES, 'UTF-8');
                        $out .= "\n   📁 <b>{$current_group_safe}:</b>\n";
                    }
                    $status_icon = $acc['active'] ? '✅' : '❌';
                    $acc_name_safe = htmlspecialchars($acc['account_name'], ENT_QUOTES, 'UTF-8');
                    $out .= "   {$num}. {$status_icon} <code>{$acc_name_safe}</code> (ID: {$acc['id']})\n";
                    $num++;
                }
            } else {
                $out .= "   <i>لا توجد حسابات مرتبطة</i>\n";
            }
            
            // الحدود
            $out .= "\n⚙️ <b>الحدود الخاصة بالبوت:</b>\n";
            if ($limits) {
                $mode_ar = [
                    'daily' => 'يومي',
                    'weekly' => 'أسبوعي',
                    'monthly' => 'شهري'
                ];
                $out .= "   • الوضع: " . ($mode_ar[$limits['mode']] ?? $limits['mode']) . "\n";
                $out .= "   • الحد اليومي: {$limits['per_day']} كود\n";
                if ($limits['weekly_cap']) $out .= "   • الحد الأسبوعي: {$limits['weekly_cap']} كود\n";
                if ($limits['monthly_cap']) $out .= "   • الحد الشهري: {$limits['monthly_cap']} كود\n";
                if ($limits['ban_days']) $out .= "   • مدة الحظر: {$limits['ban_days']} يوم\n";
            } else {
                $out .= "   <i>يستخدم الحدود العامة</i>\n";
            }
            
            $send_result = tg_send($CONTROL_TOKEN, $chat_id, $out);
            if (!($send_result['ok'] ?? false)) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ فشل إرسال التفاصيل. الخطأ:\n" .
                    "<code>" . htmlspecialchars($send_result['description'] ?? 'unknown', ENT_QUOTES, 'UTF-8') . "</code>\n\n" .
                    "طول الرسالة: " . strlen($out) . " حرف"
                );
            }
            break;
        }

        case '/account_details': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ <b>الاستخدام:</b>\n" .
                    "<code>/account_details &lt;account_name أو ID&gt;</code>\n\n" .
                    "💡 <i>مثال:</i> <code>/account_details steamuser123</code>\n" .
                    "أو: <code>/account_details 42</code>");
                break;
            }
            
            $search = $args[0];
            
            // البحث بالاسم أو ID
            if (is_numeric($search)) {
                $stmt = $db->prepare("SELECT * FROM steam_accounts WHERE id=?");
                $stmt->execute([(int)$search]);
            } else {
                $stmt = $db->prepare("SELECT * FROM steam_accounts WHERE account_name=?");
                $stmt->execute([$search]);
            }
            
            $account = $stmt->fetch();
            
            if (!$account) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحساب <b>{$search}</b> غير موجود");
                break;
            }
            
            // الحصول على المجموعة
            $stmt = $db->prepare("
                SELECT ag.*
                FROM group_accounts ga
                JOIN account_groups ag ON ag.id = ga.group_id
                WHERE ga.account_id = ?
                LIMIT 1
            ");
            $stmt->execute([$account['id']]);
            $group = $stmt->fetch();
            
            // الحصول على البوتات المرتبطة (عبر المجموعة)
            $bots = [];
            if ($group) {
                $stmt = $db->prepare("
                    SELECT sb.id, sb.label, sb.is_paused
                    FROM bot_groups bg
                    JOIN sub_bots sb ON sb.id = bg.bot_id
                    WHERE bg.group_id = ?
                    ORDER BY sb.label ASC
                ");
                $stmt->execute([$group['id']]);
                $bots = $stmt->fetchAll();
            }
            
            // الحصول على الحدود الخاصة بالحساب
            $stmt = $db->prepare("SELECT * FROM limits_override WHERE account_id=?");
            $stmt->execute([$account['id']]);
            $override_limits = $stmt->fetch();
            
            // بناء الرد
            $status = $account['active'] ? '✅ نشط' : '❌ معطل';
            $acc_name_safe = htmlspecialchars($account['account_name'], ENT_QUOTES, 'UTF-8');
            
            // عرض Steam Guard Secret بشكل آمن
            $secret = $account['shared_secret'] ?? '';
            if (!empty($secret)) {
                $secret_display = strlen($secret) > 10 ? substr($secret, 0, 10) . '...' : $secret;
            } else {
                $secret_display = 'غير متوفر';
            }
            $secret_safe = htmlspecialchars($secret_display, ENT_QUOTES, 'UTF-8');
            
            $out = "👤 <b>معلومات الحساب: {$acc_name_safe}</b>\n\n";
            $out .= "🔹 <b>ID:</b> {$account['id']}\n";
            $out .= "🔹 <b>الحالة:</b> {$status}\n";
            $out .= "🔹 <b>Steam Guard:</b> <code>{$secret_safe}</code>\n\n";
            
            // المجموعة
            $out .= "📂 <b>المجموعة المرتبطة:</b>\n";
            if ($group) {
                $status_icon = $group['active'] ? '✅' : '⛔';
                $group_name_safe = htmlspecialchars($group['group_name'], ENT_QUOTES, 'UTF-8');
                $out .= "   {$status_icon} <code>{$group_name_safe}</code> (ID: {$group['id']})\n";
            } else {
                $out .= "   <i>لا توجد مجموعة مرتبطة</i>\n";
            }
            
            // البوتات
            $out .= "\n🤖 <b>البوتات المرتبطة:</b> (" . count($bots) . ")\n";
            if ($bots) {
                foreach ($bots as $i => $bot) {
                    $num = $i + 1;
                    $status_icon = $bot['is_paused'] ? '⏸️' : '▶️';
                    $bot_label_safe = htmlspecialchars($bot['label'], ENT_QUOTES, 'UTF-8');
                    $out .= "   {$num}. {$status_icon} <code>{$bot_label_safe}</code> (ID: {$bot['id']})\n";
                }
            } else {
                $out .= "   <i>لا توجد بوتات مرتبطة</i>\n";
            }
            
            // الحدود
            $out .= "\n⚙️ <b>الحدود الخاصة:</b>\n";
            if ($override_limits) {
                $mode_ar = [
                    'daily' => 'يومي',
                    'weekly' => 'أسبوعي',
                    'monthly' => 'شهري'
                ];
                $out .= "   ✅ <b>حدود خاصة بالحساب:</b>\n";
                $out .= "   • الوضع: " . ($mode_ar[$override_limits['mode']] ?? $override_limits['mode']) . "\n";
                $out .= "   • الحد اليومي: {$override_limits['per_day']} كود\n";
                if ($override_limits['cap']) $out .= "   • السقف: {$override_limits['cap']} كود\n";
                if ($override_limits['ban_days']) $out .= "   • مدة الحظر: {$override_limits['ban_days']} يوم\n";
            } else {
                $out .= "   <i>يستخدم حدود البوت أو الحدود العامة</i>\n";
                $out .= "   <i>الأولوية: override → bot → global</i>\n";
            }
            
            $send_result = tg_send($CONTROL_TOKEN, $chat_id, $out);
            if (!($send_result['ok'] ?? false)) {
                tg_send($CONTROL_TOKEN, $chat_id, 
                    "❌ فشل إرسال التفاصيل. الخطأ:\n" .
                    "<code>" . htmlspecialchars($send_result['description'] ?? 'unknown', ENT_QUOTES, 'UTF-8') . "</code>\n\n" .
                    "طول الرسالة: " . strlen($out) . " حرف"
                );
            }
            break;
        }
        
        // ============================================
        // أوامر حدود المستخدمين (User-Level Limits)
        // ============================================
        
        case '/set_user_limit': {
            if (count($args) < 3) {
                $usage = "❌ <b>الاستخدام:</b>\n\n";
                $usage .= "<code>/set_user_limit &lt;user_id&gt; &lt;mode&gt; &lt;per_day&gt; [cap] [ban_days] [bot]</code>\n\n";
                $usage .= "📊 <b>الأوضاع:</b>\n";
                $usage .= "• <code>daily</code> - حد يومي\n";
                $usage .= "• <code>weekly</code> - يومي + أسبوعي\n";
                $usage .= "• <code>monthly</code> - يومي + شهري\n\n";
                $usage .= "💡 <b>أمثلة:</b>\n";
                $usage .= "• <code>/set_user_limit 123456 daily 10</code>\n";
                $usage .= "• <code>/set_user_limit 123456 weekly 5 20 7</code>\n";
                $usage .= "• <code>/set_user_limit 123456 monthly 3 50 30 test1</code>";
                tg_send($CONTROL_TOKEN, $chat_id, $usage);
                break;
            }
            
            $target_user = (int)$args[0];
            $mode = strtolower($args[1]);
            $per_day = (int)$args[2];
            $cap = isset($args[3]) ? (int)$args[3] : null;
            $ban_days = isset($args[4]) ? (int)$args[4] : null;
            $bot_label = $args[5] ?? null;
            
            if (!in_array($mode, ['daily', 'weekly', 'monthly'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الوضع يجب أن يكون: daily, weekly, أو monthly");
                break;
            }
            
            if ($per_day <= 0) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ الحد اليومي يجب أن يكون أكبر من صفر");
                break;
            }
            
            if ($bot_label) {
                $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
                $stmt->execute([$bot_label]);
                if (!$stmt->fetch()) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$bot_label}</b> غير موجود");
                    break;
                }
            }
            
            try {
                $table_exists = $db->query("SHOW TABLES LIKE 'limits_user'")->fetch();
                if (!$table_exists) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ جدول حدود المستخدمين غير موجود!\n\nنفّذ: <code>sql_updates/02_user_limits.sql</code>");
                    break;
                }
                
                if ($mode === 'weekly') {
                    $stmt = $db->prepare("INSERT INTO limits_user(user_id, bot_label, mode, per_day, weekly_cap, ban_days) VALUES(?,?,?,?,?,?) ON DUPLICATE KEY UPDATE mode=VALUES(mode), per_day=VALUES(per_day), weekly_cap=VALUES(weekly_cap), ban_days=VALUES(ban_days), updated_at=NOW()");
                    $stmt->execute([$target_user, $bot_label, $mode, $per_day, $cap, $ban_days]);
                } elseif ($mode === 'monthly') {
                    $stmt = $db->prepare("INSERT INTO limits_user(user_id, bot_label, mode, per_day, monthly_cap, ban_days) VALUES(?,?,?,?,?,?) ON DUPLICATE KEY UPDATE mode=VALUES(mode), per_day=VALUES(per_day), monthly_cap=VALUES(monthly_cap), ban_days=VALUES(ban_days), updated_at=NOW()");
                    $stmt->execute([$target_user, $bot_label, $mode, $per_day, $cap, $ban_days]);
                } else {
                    $stmt = $db->prepare("INSERT INTO limits_user(user_id, bot_label, mode, per_day, ban_days) VALUES(?,?,?,?,?) ON DUPLICATE KEY UPDATE mode=VALUES(mode), per_day=VALUES(per_day), ban_days=VALUES(ban_days), updated_at=NOW()");
                    $stmt->execute([$target_user, $bot_label, $mode, $per_day, $ban_days]);
                }
                
                $scope = $bot_label ? "في <b>{$bot_label}</b>" : "لجميع البوتات";
                $out = "✅ <b>تم تعيين حدود المستخدم</b>\n\n";
                $out .= "👤 ID: <code>{$target_user}</code>\n";
                $out .= "🎯 النطاق: {$scope}\n";
                $out .= "📊 الوضع: {$mode}\n";
                $out .= "📅 يومي: {$per_day}\n";
                if ($cap) $out .= "📈 الحد الكلي: {$cap}\n";
                if ($ban_days) $out .= "🚫 الحظر: {$ban_days} يوم\n";
                
                tg_send($CONTROL_TOKEN, $chat_id, $out);
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        case '/clear_user_limit': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/clear_user_limit &lt;user_id&gt; [bot]</code>");
                break;
            }
            
            $target_user = (int)$args[0];
            $bot_label = $args[1] ?? null;
            
            try {
                if ($bot_label) {
                    $stmt = $db->prepare("DELETE FROM limits_user WHERE user_id=? AND bot_label=?");
                    $stmt->execute([$target_user, $bot_label]);
                    $scope = "من <b>{$bot_label}</b>";
                } else {
                    $stmt = $db->prepare("DELETE FROM limits_user WHERE user_id=? AND bot_label IS NULL");
                    $stmt->execute([$target_user]);
                    $scope = "العامة";
                }
                
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إزالة حدود <code>{$target_user}</code> {$scope}");
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        case '/show_user_limit': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/show_user_limit &lt;user_id&gt;</code>");
                break;
            }
            
            $target_user = (int)$args[0];
            
            try {
                $stmt = $db->prepare("SELECT * FROM limits_user WHERE user_id=? ORDER BY bot_label");
                $stmt->execute([$target_user]);
                $limits = $stmt->fetchAll();
                
                if (empty($limits)) {
                    tg_send($CONTROL_TOKEN, $chat_id, "لا توجد حدود للمستخدم <code>{$target_user}</code>");
                    break;
                }
                
                $out = "📊 <b>حدود المستخدم {$target_user}:</b>\n\n";
                foreach ($limits as $lim) {
                    $scope = $lim['bot_label'] ? "<code>{$lim['bot_label']}</code>" : "عام";
                    $out .= "🎯 {$scope}\n   └ {$lim['mode']} | يومي: {$lim['per_day']}";
                    if ($lim['weekly_cap']) $out .= " | أسبوعي: {$lim['weekly_cap']}";
                    if ($lim['monthly_cap']) $out .= " | شهري: {$lim['monthly_cap']}";
                    $out .= "\n\n";
                }
                
                tg_send($CONTROL_TOKEN, $chat_id, $out);
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        // ============================================
        // أوامر Webhook
        // ============================================
        
        case '/webhook_create':
        case '/webhook_enable': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>{$cmd} &lt;bot_label&gt;</code>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
                break;
            }
            
            $token = $bot['bot_token'];
            $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($token);
            
            if ($cmd === '/webhook_create') {
                $out = "🔗 <b>إعداد Webhook للبوت {$label}:</b>\n\n";
                $out .= "📌 الرابط:\n<code>{$webhookUrl}</code>\n\n";
                $out .= "⚡ تفعيل سريع:\nhttps://api.telegram.org/bot{$token}/setWebhook?url=" . urlencode($webhookUrl);
                tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            } else {
                $result = tg_call($token, 'setWebhook', ['url' => $webhookUrl]);
                if (!empty($result['ok'])) {
                    tg_send($CONTROL_TOKEN, $chat_id, "✅ تم تفعيل Webhook للبوت <b>{$label}</b>\n\n🔗 {$webhookUrl}");
                } else {
                    $error = $result['description'] ?? json_encode($result);
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل التفعيل:\n{$error}");
                }
            }
            break;
        }
        
        case '/webhook_reset': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/webhook_reset &lt;bot_label&gt;</code>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
                break;
            }
            
            $token = $bot['bot_token'];
            tg_call($token, 'deleteWebhook', ['drop_pending_updates' => true]);
            usleep(500000);
            
            $webhookUrl = $DOMAIN . "/sub_bot.php?bot=" . urlencode($token);
            $result = tg_call($token, 'setWebhook', ['url' => $webhookUrl]);
            
            if (!empty($result['ok'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إعادة ضبط Webhook للبوت <b>{$label}</b>");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ فشلت إعادة الضبط:\n" . ($result['description'] ?? json_encode($result)));
            }
            break;
        }
        
        case '/webhook_delete': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/webhook_delete &lt;bot_label&gt;</code>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
                break;
            }
            
            $token = $bot['bot_token'];
            $result = tg_call($token, 'deleteWebhook', ['drop_pending_updates' => true]);
            
            if (!empty($result['ok'])) {
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف Webhook للبوت <b>{$label}</b>");
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل الحذف:\n" . ($result['description'] ?? json_encode($result)));
            }
            break;
        }
        
        case '/webhook_info': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/webhook_info &lt;bot_label&gt;</code>");
                break;
            }
            
            $label = $args[0];
            $stmt = $db->prepare("SELECT bot_token FROM sub_bots WHERE label=?");
            $stmt->execute([$label]);
            $bot = $stmt->fetch();
            
            if (!$bot) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
                break;
            }
            
            $token = $bot['bot_token'];
            $result = tg_call($token, 'getWebhookInfo', []);
            
            if (!empty($result['ok'])) {
                $info = $result['result'];
                $out = "ℹ️ <b>معلومات Webhook - {$label}:</b>\n\n";
                $out .= "🔗 URL: " . ($info['url'] ?: 'غير محدد') . "\n";
                $out .= "⏳ التحديثات المنتظرة: " . ($info['pending_update_count'] ?? 0) . "\n";
                if (!empty($info['last_error_message'])) {
                    $out .= "\n❌ آخر خطأ:\n{$info['last_error_message']}\n";
                    $out .= "📅 {" . date('Y-m-d H:i:s', $info['last_error_date']) . "}\n";
                }
                $out .= "\n🔢 الحد الأقصى: " . ($info['max_connections'] ?? 40);
                tg_send($CONTROL_TOKEN, $chat_id, $out);
            } else {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ فشل جلب المعلومات");
            }
            break;
        }
        
        // ============================================
        // أوامر الإحصاءات والتقارير
        // ============================================
        
        case '/stats_bots': {
            $total = $db->query("SELECT COUNT(*) as cnt FROM sub_bots")->fetch()['cnt'];
            $active = $db->query("SELECT COUNT(*) as cnt FROM sub_bots WHERE is_paused=0")->fetch()['cnt'];
            $paused = $total - $active;
            
            $out = "📊 <b>إحصاءات البوتات:</b>\n\n";
            $out .= "🤖 الإجمالي: <b>{$total}</b>\n";
            $out .= "▶️ نشطة: <b>{$active}</b>\n";
            $out .= "⏸️ موقوفة: <b>{$paused}</b>\n\n";
            
            $top = $db->query("SELECT b.label, COUNT(DISTINCT u.user_id) as cnt FROM sub_bots b LEFT JOIN allowed_users u ON u.bot_token=b.bot_token GROUP BY b.id ORDER BY cnt DESC LIMIT 5")->fetchAll();
            
            if (!empty($top)) {
                $out .= "👥 <b>الأكثر مستخدمين:</b>\n";
                foreach ($top as $t) {
                    $out .= "• <code>{$t['label']}</code> - {$t['cnt']} مستخدم\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/stats_users': {
            $total = $db->query("SELECT COUNT(DISTINCT user_id) as cnt FROM allowed_users")->fetch()['cnt'];
            $banned = $db->query("SELECT COUNT(DISTINCT user_id) as cnt FROM allowed_users WHERE banned=1")->fetch()['cnt'];
            $active = $total - $banned;
            
            $out = "📊 <b>إحصاءات المستخدمين:</b>\n\n";
            $out .= "👥 الإجمالي: <b>{$total}</b>\n";
            $out .= "✅ نشطون: <b>{$active}</b>\n";
            $out .= "🚫 محظورون: <b>{$banned}</b>\n\n";
            
            $top = $db->query("SELECT user_id, COUNT(*) as cnt FROM steam_requests WHERE request_time >= DATE_SUB(NOW(), INTERVAL 7 DAY) GROUP BY user_id ORDER BY cnt DESC LIMIT 5")->fetchAll();
            
            if (!empty($top)) {
                $out .= "🔥 <b>الأكثر نشاطاً (7 أيام):</b>\n";
                foreach ($top as $t) {
                    $out .= "• <code>{$t['user_id']}</code> - {$t['cnt']} طلب\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/stats_groups': {
            $total = $db->query("SELECT COUNT(*) as cnt FROM account_groups")->fetch()['cnt'];
            $active = $db->query("SELECT COUNT(*) as cnt FROM account_groups WHERE active=1")->fetch()['cnt'];
            
            $out = "📊 <b>إحصاءات المجموعات:</b>\n\n";
            $out .= "📂 الإجمالي: <b>{$total}</b>\n";
            $out .= "✅ نشطة: <b>{$active}</b>\n\n";
            
            $top = $db->query("SELECT g.group_name, COUNT(DISTINCT ga.account_id) as acc_cnt, COUNT(DISTINCT bg.bot_id) as bot_cnt FROM account_groups g LEFT JOIN group_accounts ga ON ga.group_id=g.id LEFT JOIN bot_groups bg ON bg.group_id=g.id WHERE g.active=1 GROUP BY g.id ORDER BY acc_cnt DESC LIMIT 5")->fetchAll();
            
            if (!empty($top)) {
                $out .= "🏆 <b>الأكبر:</b>\n";
                foreach ($top as $t) {
                    $out .= "• <b>{$t['group_name']}</b> - {$t['acc_cnt']} حساب, {$t['bot_cnt']} بوت\n";
                }
            }
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/stats_vouchers': {
            $total = $db->query("SELECT COUNT(*) as cnt FROM vouchers")->fetch()['cnt'];
            $used = $db->query("SELECT COUNT(*) as cnt FROM vouchers WHERE current_uses >= max_uses")->fetch()['cnt'];
            $available = $total - $used;
            $redemptions = $db->query("SELECT COUNT(*) as cnt FROM voucher_redemptions")->fetch()['cnt'];
            
            $out = "📊 <b>إحصاءات القسائم:</b>\n\n";
            $out .= "🎫 الإجمالي: <b>{$total}</b>\n";
            $out .= "✅ متاحة: <b>{$available}</b>\n";
            $out .= "❌ مستهلكة: <b>{$used}</b>\n";
            $out .= "📈 التفعيلات: <b>{$redemptions}</b>\n";
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/stats_broadcast': {
            $total = $db->query("SELECT COUNT(*) as cnt FROM broadcast_log")->fetch()['cnt'];
            $recipients = $db->query("SELECT SUM(recipients_count) as total FROM broadcast_log")->fetch()['total'] ?? 0;
            
            $out = "📊 <b>إحصاءات البث:</b>\n\n";
            $out .= "📢 عمليات البث: <b>{$total}</b>\n";
            $out .= "👥 المستلمون: <b>{$recipients}</b>\n\n";
            
            $recent = $db->query("SELECT b.label, bl.recipients_count, bl.created_at FROM broadcast_log bl LEFT JOIN sub_bots b ON b.bot_token=bl.bot_token ORDER BY bl.id DESC LIMIT 5")->fetchAll();
            
            if (!empty($recent)) {
                $out .= "📋 <b>الأحدث:</b>\n";
                foreach ($recent as $r) {
                    $bot_name = $r['label'] ?? 'غير محدد';
                    $date = date('m/d H:i', strtotime($r['created_at']));
                    $out .= "• {$bot_name} - {$r['recipients_count']} ({$date})\n";
                }
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        case '/stats_all': {
            $out = "📊 <b>ملخص النظام:</b>\n\n";
            $out .= "🤖 البوتات: <b>" . $db->query("SELECT COUNT(*) as c FROM sub_bots")->fetch()['c'] . "</b>\n";
            $out .= "👥 المستخدمون: <b>" . $db->query("SELECT COUNT(DISTINCT user_id) as c FROM allowed_users")->fetch()['c'] . "</b>\n";
            $out .= "🎮 الحسابات: <b>" . $db->query("SELECT COUNT(*) as c FROM steam_accounts WHERE active=1")->fetch()['c'] . "</b>\n";
            $out .= "📂 المجموعات: <b>" . $db->query("SELECT COUNT(*) as c FROM account_groups WHERE active=1")->fetch()['c'] . "</b>\n";
            $out .= "🎫 قسائم متاحة: <b>" . $db->query("SELECT COUNT(*) as c FROM vouchers WHERE current_uses < max_uses")->fetch()['c'] . "</b>\n\n";
            $out .= "📈 طلبات اليوم: <b>" . $db->query("SELECT COUNT(*) as c FROM steam_requests WHERE DATE(request_time) = CURDATE()")->fetch()['c'] . "</b>\n";
            
            tg_send($CONTROL_TOKEN, $chat_id, $out);
            break;
        }
        
        // ============================================
        // أوامر نظام الردود الموسع
        // ============================================
        
        case '/response_add':
        case '/response_edit': {
            if (count($args) < 2) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>{$cmd} &lt;key&gt; &lt;text&gt;</code>");
                break;
            }
            
            $key = $args[0];
            $text = trim(substr($message_text, strlen("{$cmd} {$key}")));
            
            if ($cmd === '/response_edit') {
                $existing = template_get($key);
                if (empty($existing)) {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ المفتاح <code>{$key}</code> غير موجود\n\nاستخدم /response_add لإنشائه");
                    break;
                }
            }
            
            try {
                template_set($key, $text);
                tg_send($CONTROL_TOKEN, $chat_id, "✅ تم " . ($cmd === '/response_add' ? 'إضافة' : 'تحديث') . " الرد <code>{$key}</code>");
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        case '/response_delete': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/response_delete &lt;key&gt;</code>");
                break;
            }
            
            $key = $args[0];
            
            try {
                $stmt = $db->prepare("DELETE FROM message_templates WHERE `key`=?");
                $stmt->execute([$key]);
                
                if ($stmt->rowCount() > 0) {
                    tg_send($CONTROL_TOKEN, $chat_id, "✅ تم حذف الرد <code>{$key}</code>");
                } else {
                    tg_send($CONTROL_TOKEN, $chat_id, "❌ المفتاح <code>{$key}</code> غير موجود");
                }
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        case '/response_list': {
            try {
                $templates = $db->query("SELECT `key` FROM message_templates ORDER BY `key`")->fetchAll();
                
                if (empty($templates)) {
                    tg_send($CONTROL_TOKEN, $chat_id, "لا توجد ردود محفوظة");
                    break;
                }
                
                $out = "📝 <b>قائمة الردود (" . count($templates) . "):</b>\n\n";
                foreach ($templates as $t) {
                    $out .= "• <code>{$t['key']}</code>\n";
                }
                
                tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            } catch (Exception $e) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ خطأ: " . $e->getMessage());
            }
            break;
        }
        
        case '/response_view': {
            if (count($args) < 1) {
                tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام: <code>/response_view &lt;key&gt;</code>");
                break;
            }
            
            $key = $args[0];
            $text = template_get($key);
            
            if (empty($text)) {
                tg_send($CONTROL_TOKEN, $chat_id, "❌ المفتاح <code>{$key}</code> غير موجود");
            } else {
                $out = "📝 <b>الرد: {$key}</b>\n\n{$text}";
                tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            }
            break;
        }
        
        // ============================================
        // عرض رسائل الترحيب
        // ============================================
        
        case '/show_welcome_messages': {
            $bots = $db->query("SELECT label, welcome_message FROM sub_bots ORDER BY label")->fetchAll();
            
            if (empty($bots)) {
                tg_send($CONTROL_TOKEN, $chat_id, "لا توجد بوتات");
                break;
            }
            
            $out = "💬 <b>رسائل الترحيب:</b>\n\n";
            foreach ($bots as $bot) {
                $out .= "🤖 <b>{$bot['label']}</b>\n";
                $welcome = $bot['welcome_message'] ?: '(لم يتم تعيينها)';
                $out .= "└ {$welcome}\n\n";
            }
            
            tg_send_long($CONTROL_TOKEN, $chat_id, $out);
            break;
        }

        // ════════════════════════════════════════════════════════════

        default:
            tg_send($CONTROL_TOKEN, $chat_id, "أمر غير معروف. استخدم /help.");
            break;
    }
    // END_COMMANDS_MARKER
}

http_response_code(200);
echo 'OK';
